/*
 * Decompiled with CFR 0.152.
 */
package org.openhab.core.internal.scheduler;

import java.time.Duration;
import java.time.Instant;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.time.temporal.TemporalAdjuster;
import java.util.Objects;
import java.util.concurrent.Callable;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Delayed;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Consumer;
import org.eclipse.jdt.annotation.NonNullByDefault;
import org.eclipse.jdt.annotation.Nullable;
import org.openhab.core.common.ThreadPoolManager;
import org.openhab.core.scheduler.ScheduledCompletableFuture;
import org.openhab.core.scheduler.Scheduler;
import org.openhab.core.scheduler.SchedulerRunnable;
import org.openhab.core.scheduler.SchedulerTemporalAdjuster;
import org.osgi.service.component.annotations.Component;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Component(service={SchedulerImpl.class}, immediate=true)
@NonNullByDefault
public class SchedulerImpl
implements Scheduler {
    private static final String SCHEDULER_THREAD_POOL = "scheduler";
    private static final int ALLOWED_DEVIATION_MILLISECONDS = 2000;
    private final Logger logger = LoggerFactory.getLogger(SchedulerImpl.class);
    private final ScheduledExecutorService executor = ThreadPoolManager.getScheduledPool("scheduler");

    @Override
    public ScheduledCompletableFuture<Instant> after(Duration duration) {
        Instant start = Instant.now();
        return this.after(() -> start, duration);
    }

    @Override
    public <T> ScheduledCompletableFuture<T> after(Callable<T> callable, Duration duration) {
        return this.afterInternal(new ScheduledCompletableFutureOnce(null, duration), callable);
    }

    private <T> ScheduledCompletableFutureOnce<T> afterInternal(ScheduledCompletableFutureOnce<T> deferred, Callable<T> callable) {
        long duration = Math.max(100L, deferred.getScheduledTime().minus(this.currentTimeMillis(), ChronoUnit.MILLIS).toInstant().toEpochMilli());
        ScheduledFuture<?> future = this.executor.schedule(() -> {
            try {
                long timeLeft = deferred.getDelay(TimeUnit.MILLISECONDS);
                if (timeLeft > 2000L) {
                    this.logger.trace("Scheduled task is re-scheduled because the scheduler ran {} milliseconds to early.", (Object)timeLeft);
                    this.afterInternal(deferred, callable);
                } else {
                    this.logger.trace("Scheduled task is run now.");
                    deferred.complete(callable.call());
                }
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
            catch (Exception e) {
                this.logger.warn("Scheduled job '{}' failed and stopped", (Object)Objects.requireNonNullElse(scheduledCompletableFutureOnce.identifier, "<unknown>"), (Object)e);
                deferred.completeExceptionally(e);
            }
        }, duration, TimeUnit.MILLISECONDS);
        if (this.logger.isTraceEnabled()) {
            this.logger.trace("Scheduled a task to run in {} seconds.", (Object)TimeUnit.MILLISECONDS.toSeconds(duration));
        }
        deferred.exceptionally(e -> {
            this.logger.trace("Scheduled task stopped with exception ", e);
            if (e instanceof CancellationException) {
                future.cancel(true);
            }
            return null;
        });
        return deferred;
    }

    @Override
    public <T> ScheduledCompletableFuture<T> before(CompletableFuture<T> promise, Duration timeout) {
        AtomicBoolean done = new AtomicBoolean();
        Consumer<Runnable> runOnce = runnable -> {
            if (!done.getAndSet(true)) {
                runnable.run();
            }
        };
        ScheduledCompletableFutureOnce wrappedPromise = new ScheduledCompletableFutureOnce(null, timeout);
        Callable<Object> callable = () -> {
            wrappedPromise.completeExceptionally(new TimeoutException());
            return null;
        };
        ScheduledCompletableFutureOnce<Object> afterPromise = this.afterInternal(wrappedPromise, callable);
        wrappedPromise.exceptionally(e -> {
            if (e instanceof CancellationException) {
                afterPromise.cancel(true);
            }
            return null;
        });
        ((CompletableFuture)promise.thenAccept(p -> runOnce.accept(() -> {
            boolean bl = wrappedPromise.complete(p);
        }))).exceptionally(ex -> {
            runOnce.accept(() -> {
                boolean bl = wrappedPromise.completeExceptionally((Throwable)ex);
            });
            return null;
        });
        return wrappedPromise;
    }

    @Override
    public ScheduledCompletableFuture<Instant> at(Instant instant) {
        return this.at(() -> instant, instant);
    }

    @Override
    public <T> ScheduledCompletableFuture<T> at(Callable<T> callable, Instant instant) {
        return this.atInternal(new ScheduledCompletableFutureOnce(null, ZonedDateTime.ofInstant(instant, ZoneId.systemDefault())), callable);
    }

    private <T> ScheduledCompletableFuture<T> atInternal(ScheduledCompletableFutureOnce<T> deferred, Callable<T> callable) {
        return this.afterInternal(deferred, callable);
    }

    @Override
    public <T> ScheduledCompletableFuture<T> schedule(SchedulerRunnable runnable, TemporalAdjuster temporalAdjuster) {
        return this.schedule(runnable, null, temporalAdjuster);
    }

    @Override
    public <T> ScheduledCompletableFuture<T> schedule(SchedulerRunnable runnable, @Nullable String identifier, TemporalAdjuster temporalAdjuster) {
        ScheduledCompletableFutureRecurring schedule = new ScheduledCompletableFutureRecurring(identifier, ZonedDateTime.now());
        this.schedule(schedule, runnable, identifier, temporalAdjuster);
        return schedule;
    }

    private <T> void schedule(ScheduledCompletableFutureRecurring<T> recurringSchedule, SchedulerRunnable runnable, @Nullable String identifier, TemporalAdjuster temporalAdjuster) {
        ZonedDateTime newTime = recurringSchedule.getScheduledTime().with(temporalAdjuster);
        ScheduledCompletableFutureOnce deferred = new ScheduledCompletableFutureOnce(identifier, ZonedDateTime.from(newTime));
        deferred.thenAccept(v -> {
            SchedulerTemporalAdjuster schedulerTemporalAdjuster;
            if (temporalAdjuster instanceof SchedulerTemporalAdjuster && !(schedulerTemporalAdjuster = (SchedulerTemporalAdjuster)temporalAdjuster).isDone(newTime)) {
                this.schedule(recurringSchedule, runnable, identifier, temporalAdjuster);
                return;
            }
            recurringSchedule.complete(v);
        });
        recurringSchedule.setScheduledPromise(deferred);
        this.atInternal(deferred, () -> {
            runnable.run();
            return null;
        });
    }

    protected long currentTimeMillis() {
        return System.currentTimeMillis();
    }

    private static class ScheduledCompletableFutureOnce<T>
    extends CompletableFuture<T>
    implements ScheduledCompletableFuture<T> {
        private ZonedDateTime scheduledTime;
        private @Nullable String identifier;

        public ScheduledCompletableFutureOnce(@Nullable String identifier, Duration duration) {
            this(identifier, ZonedDateTime.now().plusNanos(duration.toNanos()));
        }

        public ScheduledCompletableFutureOnce(@Nullable String identifier, ZonedDateTime scheduledTime) {
            this.identifier = identifier;
            this.scheduledTime = scheduledTime;
        }

        @Override
        public CompletableFuture<T> getPromise() {
            return this;
        }

        @Override
        public long getDelay(@Nullable TimeUnit timeUnit) {
            ZonedDateTime scheduledTime = this.scheduledTime;
            if (timeUnit == null) {
                return 0L;
            }
            long remaining = scheduledTime.toInstant().toEpochMilli() - System.currentTimeMillis();
            return timeUnit.convert(remaining, TimeUnit.MILLISECONDS);
        }

        @Override
        public int compareTo(@Nullable Delayed timeUnit) {
            return timeUnit == null ? -1 : Long.compare(this.getDelay(TimeUnit.MILLISECONDS), timeUnit.getDelay(TimeUnit.MILLISECONDS));
        }

        @Override
        public ZonedDateTime getScheduledTime() {
            return this.scheduledTime;
        }
    }

    private static class ScheduledCompletableFutureRecurring<T>
    extends ScheduledCompletableFutureOnce<T> {
        private volatile @Nullable ScheduledCompletableFuture<T> scheduledPromise;

        public ScheduledCompletableFutureRecurring(@Nullable String identifier, ZonedDateTime scheduledTime) {
            super(identifier, scheduledTime);
            this.exceptionally((T e) -> {
                ScheduledCompletableFutureRecurring scheduledCompletableFutureRecurring = this;
                synchronized (scheduledCompletableFutureRecurring) {
                    ScheduledCompletableFuture<T> scheduledCompletableFuture;
                    if (e instanceof CancellationException && (scheduledCompletableFuture = this.scheduledPromise) instanceof ScheduledCompletableFuture) {
                        ScheduledCompletableFuture<T> promise = scheduledCompletableFuture;
                        promise.cancel(true);
                    }
                }
                return null;
            });
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void setScheduledPromise(ScheduledCompletableFuture<T> future) {
            ScheduledCompletableFutureRecurring scheduledCompletableFutureRecurring = this;
            synchronized (scheduledCompletableFutureRecurring) {
                if (this.isCancelled()) {
                    future.cancel(true);
                } else {
                    this.scheduledPromise = future;
                    future.getPromise().exceptionally((T ex) -> {
                        this.completeExceptionally((Throwable)ex);
                        return null;
                    });
                }
            }
        }

        @Override
        public long getDelay(@Nullable TimeUnit timeUnit) {
            long l;
            ScheduledCompletableFuture<T> scheduledCompletableFuture = this.scheduledPromise;
            if (scheduledCompletableFuture instanceof ScheduledCompletableFuture) {
                ScheduledCompletableFuture<T> promise = scheduledCompletableFuture;
                l = promise.getDelay(timeUnit);
            } else {
                l = 0L;
            }
            return l;
        }

        @Override
        public ZonedDateTime getScheduledTime() {
            ZonedDateTime zonedDateTime;
            ScheduledCompletableFuture<T> scheduledCompletableFuture = this.scheduledPromise;
            if (scheduledCompletableFuture instanceof ScheduledCompletableFuture) {
                ScheduledCompletableFuture<T> promise = scheduledCompletableFuture;
                zonedDateTime = promise.getScheduledTime();
            } else {
                zonedDateTime = super.getScheduledTime();
            }
            return zonedDateTime;
        }
    }
}

