// SPDX-License-Identifier: GPL-2.0
/*
 * MediaTek clock driver for MT8183 SoC
 *
 * Copyright (C) 2020 BayLibre, SAS
 * Copyright (c) 2020 MediaTek Inc.
 * Author: Fabien Parent <fparent@baylibre.com>
 * Author: Weiyi Lu <weiyi.lu@mediatek.com>
 */

#include <common.h>
#include <dm.h>
#include <asm/io.h>
#include <dt-bindings/clock/mt8183-clk.h>

#include "clk-mtk.h"

#define MT8183_PLL_FMAX		(3800UL * MHZ)
#define MT8183_PLL_FMIN		(1500UL * MHZ)

/* apmixedsys */
#define PLL(_id, _reg, _pwr_reg, _en_mask, _flags, _rst_bar_mask, _pcwbits, \
	    _pcwibits, _pd_reg, _pd_shift, _pcw_reg, _pcw_shift) {	\
		.id = _id,						\
		.reg = _reg,						\
		.pwr_reg = _pwr_reg,					\
		.en_mask = _en_mask,					\
		.rst_bar_mask = _rst_bar_mask,				\
		.fmax = MT8183_PLL_FMAX,				\
		.fmin = MT8183_PLL_FMIN,				\
		.flags = _flags,					\
		.pcwbits = _pcwbits,					\
		.pcwibits = _pcwibits,					\
		.pd_reg = _pd_reg,					\
		.pd_shift = _pd_shift,					\
		.pcw_reg = _pcw_reg,					\
		.pcw_shift = _pcw_shift,				\
	}

static const struct mtk_pll_data apmixed_plls[] = {
	PLL(CLK_APMIXED_ARMPLL_LL, 0x0200, 0x020C, 0x00000001,
	    HAVE_RST_BAR, BIT(24), 22, 8, 0x0204, 24,
	    0x0204, 0),
	PLL(CLK_APMIXED_ARMPLL_L, 0x0210, 0x021C, 0x00000001,
	    HAVE_RST_BAR, BIT(24), 22, 8, 0x0214, 24,
	    0x0214, 0),
	PLL(CLK_APMIXED_CCIPLL, 0x0290, 0x029C, 0x00000001,
	    HAVE_RST_BAR, BIT(24), 22, 8, 0x0294, 24,
	    0x0294, 0),
	PLL(CLK_APMIXED_MAINPLL, 0x0220, 0x022C, 0x00000001,
	    HAVE_RST_BAR, BIT(24), 22, 8, 0x0224, 24,
	    0x0224, 0),
	PLL(CLK_APMIXED_UNIV2PLL, 0x0230, 0x023C, 0x00000001,
	    HAVE_RST_BAR, BIT(24), 22, 8, 0x0234, 24,
	    0x0234, 0),
	PLL(CLK_APMIXED_MSDCPLL, 0x0250, 0x025C, 0x00000001,
	    0, 0, 22, 8, 0x0254, 24, 0x0254, 0),
	PLL(CLK_APMIXED_MMPLL, 0x0270, 0x027C, 0x00000001,
	    HAVE_RST_BAR, BIT(23), 22, 8, 0x0274, 24,
	    0x0274, 0),
	PLL(CLK_APMIXED_MFGPLL, 0x0240, 0x024C, 0x00000001,
	    0, 0, 22, 8, 0x0244, 24, 0x0244, 0),
	PLL(CLK_APMIXED_TVDPLL, 0x0260, 0x026C, 0x00000001,
	    0, 0, 22, 8, 0x0264, 24, 0x0264, 0),
	PLL(CLK_APMIXED_APLL1, 0x02A0, 0x02B0, 0x00000001,
	    0, 0, 32, 8, 0x02A0, 1, 0x02A4, 0),
	PLL(CLK_APMIXED_APLL2, 0x02b4, 0x02c4, 0x00000001,
	    0, 0, 32, 8, 0x02B4, 1, 0x02B8, 0),
};

static const struct mtk_fixed_clk top_fixed_clks[] = {
	FIXED_CLK(CLK_TOP_CLK26M, CLK_XTAL, 26000000),
	FIXED_CLK(CLK_TOP_ULPOSC, CLK_XTAL, 250000),
	FIXED_CLK(CLK_TOP_UNIVP_192M, CLK_TOP_UNIVPLL, 192000000),
};

static const struct mtk_fixed_factor top_fixed_divs[] = {
	FACTOR(CLK_TOP_CLK13M, CLK_TOP_CLK26M, 1, 2, CLK_PARENT_TOPCKGEN),
	FACTOR(CLK_TOP_F26M_CK_D2, CLK_TOP_CLK26M, 1, 2, CLK_PARENT_TOPCKGEN),
	FACTOR(CLK_TOP_SYSPLL_CK, CLK_APMIXED_MAINPLL, 1,
	       1, CLK_PARENT_APMIXED),
	FACTOR(CLK_TOP_SYSPLL_D2, CLK_TOP_SYSPLL_CK, 1,
	       2, CLK_PARENT_TOPCKGEN),
	FACTOR(CLK_TOP_SYSPLL_D3, CLK_APMIXED_MAINPLL, 1,
	       3, CLK_PARENT_APMIXED),
	FACTOR(CLK_TOP_SYSPLL_D5, CLK_APMIXED_MAINPLL, 1,
	       5, CLK_PARENT_APMIXED),
	FACTOR(CLK_TOP_SYSPLL_D7, CLK_APMIXED_MAINPLL, 1,
	       7, CLK_PARENT_APMIXED),
	FACTOR(CLK_TOP_SYSPLL_D2_D2, CLK_TOP_SYSPLL_D2, 1,
	       2, CLK_PARENT_TOPCKGEN),
	FACTOR(CLK_TOP_SYSPLL_D2_D4, CLK_TOP_SYSPLL_D2, 1,
	       4, CLK_PARENT_TOPCKGEN),
	FACTOR(CLK_TOP_SYSPLL_D2_D8, CLK_TOP_SYSPLL_D2, 1,
	       8, CLK_PARENT_TOPCKGEN),
	FACTOR(CLK_TOP_SYSPLL_D2_D16, CLK_TOP_SYSPLL_D2, 1,
	       16, CLK_PARENT_TOPCKGEN),
	FACTOR(CLK_TOP_SYSPLL_D3_D2, CLK_TOP_SYSPLL_D3, 1,
	       2, CLK_PARENT_TOPCKGEN),
	FACTOR(CLK_TOP_SYSPLL_D3_D4, CLK_TOP_SYSPLL_D3, 1,
	       4, CLK_PARENT_TOPCKGEN),
	FACTOR(CLK_TOP_SYSPLL_D3_D8, CLK_TOP_SYSPLL_D3, 1,
	       8, CLK_PARENT_TOPCKGEN),
	FACTOR(CLK_TOP_SYSPLL_D5_D2, CLK_TOP_SYSPLL_D5, 1,
	       2, CLK_PARENT_TOPCKGEN),
	FACTOR(CLK_TOP_SYSPLL_D5_D4, CLK_TOP_SYSPLL_D5, 1,
	       4, CLK_PARENT_TOPCKGEN),
	FACTOR(CLK_TOP_SYSPLL_D7_D2, CLK_TOP_SYSPLL_D7, 1,
	       2, CLK_PARENT_TOPCKGEN),
	FACTOR(CLK_TOP_SYSPLL_D7_D4, CLK_TOP_SYSPLL_D7, 1,
	       4, CLK_PARENT_TOPCKGEN),
	FACTOR(CLK_TOP_UNIVPLL_CK, CLK_TOP_UNIVPLL, 1, 1, CLK_PARENT_TOPCKGEN),
	FACTOR(CLK_TOP_UNIVPLL_D2, CLK_TOP_UNIVPLL_CK, 1,
	       2, CLK_PARENT_TOPCKGEN),
	FACTOR(CLK_TOP_UNIVPLL_D3, CLK_TOP_UNIVPLL, 1, 3, CLK_PARENT_TOPCKGEN),
	FACTOR(CLK_TOP_UNIVPLL_D5, CLK_TOP_UNIVPLL, 1, 5, CLK_PARENT_TOPCKGEN),
	FACTOR(CLK_TOP_UNIVPLL_D7, CLK_TOP_UNIVPLL, 1, 7, CLK_PARENT_TOPCKGEN),
	FACTOR(CLK_TOP_UNIVPLL_D2_D2, CLK_TOP_UNIVPLL_D2, 1,
	       2, CLK_PARENT_TOPCKGEN),
	FACTOR(CLK_TOP_UNIVPLL_D2_D4, CLK_TOP_UNIVPLL_D2, 1,
	       4, CLK_PARENT_TOPCKGEN),
	FACTOR(CLK_TOP_UNIVPLL_D2_D8, CLK_TOP_UNIVPLL_D2, 1,
	       8, CLK_PARENT_TOPCKGEN),
	FACTOR(CLK_TOP_UNIVPLL_D3_D2, CLK_TOP_UNIVPLL_D3, 1,
	       2, CLK_PARENT_TOPCKGEN),
	FACTOR(CLK_TOP_UNIVPLL_D3_D4, CLK_TOP_UNIVPLL_D3, 1,
	       4, CLK_PARENT_TOPCKGEN),
	FACTOR(CLK_TOP_UNIVPLL_D3_D8, CLK_TOP_UNIVPLL_D3, 1,
	       8, CLK_PARENT_TOPCKGEN),
	FACTOR(CLK_TOP_UNIVPLL_D5_D2, CLK_TOP_UNIVPLL_D5, 1,
	       2, CLK_PARENT_TOPCKGEN),
	FACTOR(CLK_TOP_UNIVPLL_D5_D4, CLK_TOP_UNIVPLL_D5, 1,
	       4, CLK_PARENT_TOPCKGEN),
	FACTOR(CLK_TOP_UNIVPLL_D5_D8, CLK_TOP_UNIVPLL_D5, 1,
	       8, CLK_PARENT_TOPCKGEN),
	FACTOR(CLK_TOP_UNIVP_192M_CK, CLK_TOP_UNIVP_192M, 1, 1,
	       CLK_PARENT_TOPCKGEN),
	FACTOR(CLK_TOP_UNIVP_192M_D2, CLK_TOP_UNIVP_192M_CK, 1,
	       2, CLK_PARENT_TOPCKGEN),
	FACTOR(CLK_TOP_UNIVP_192M_D4, CLK_TOP_UNIVP_192M_CK, 1,
	       4, CLK_PARENT_TOPCKGEN),
	FACTOR(CLK_TOP_UNIVP_192M_D8, CLK_TOP_UNIVP_192M_CK, 1,
	       8, CLK_PARENT_TOPCKGEN),
	FACTOR(CLK_TOP_UNIVP_192M_D16, CLK_TOP_UNIVP_192M_CK, 1,
	       16, CLK_PARENT_TOPCKGEN),
	FACTOR(CLK_TOP_UNIVP_192M_D32, CLK_TOP_UNIVP_192M_CK, 1,
	       32, CLK_PARENT_TOPCKGEN),
	FACTOR(CLK_TOP_APLL1_CK, CLK_APMIXED_APLL1, 1, 1, CLK_PARENT_APMIXED),
	FACTOR(CLK_TOP_APLL1_D2, CLK_APMIXED_APLL1, 1, 2, CLK_PARENT_APMIXED),
	FACTOR(CLK_TOP_APLL1_D4, CLK_APMIXED_APLL1, 1, 4, CLK_PARENT_APMIXED),
	FACTOR(CLK_TOP_APLL1_D8, CLK_APMIXED_APLL1, 1, 8, CLK_PARENT_APMIXED),
	FACTOR(CLK_TOP_APLL2_CK, CLK_APMIXED_APLL2, 1, 1, CLK_PARENT_APMIXED),
	FACTOR(CLK_TOP_APLL2_D2, CLK_APMIXED_APLL2, 1, 2, CLK_PARENT_APMIXED),
	FACTOR(CLK_TOP_APLL2_D4, CLK_APMIXED_APLL2, 1, 4, CLK_PARENT_APMIXED),
	FACTOR(CLK_TOP_APLL2_D8, CLK_APMIXED_APLL2, 1, 8, CLK_PARENT_APMIXED),
	FACTOR(CLK_TOP_TVDPLL_CK, CLK_APMIXED_TVDPLL, 1, 1, CLK_PARENT_APMIXED),
	FACTOR(CLK_TOP_TVDPLL_D2, CLK_TOP_TVDPLL_CK, 1, 2, CLK_PARENT_TOPCKGEN),
	FACTOR(CLK_TOP_TVDPLL_D4, CLK_APMIXED_TVDPLL, 1, 4, CLK_PARENT_APMIXED),
	FACTOR(CLK_TOP_TVDPLL_D8, CLK_APMIXED_TVDPLL, 1, 8, CLK_PARENT_APMIXED),
	FACTOR(CLK_TOP_TVDPLL_D16, CLK_APMIXED_TVDPLL, 1,
	       16, CLK_PARENT_APMIXED),
	FACTOR(CLK_TOP_MMPLL_CK, CLK_APMIXED_MMPLL, 1, 1, CLK_PARENT_APMIXED),
	FACTOR(CLK_TOP_MMPLL_D4, CLK_APMIXED_MMPLL, 1, 4, CLK_PARENT_APMIXED),
	FACTOR(CLK_TOP_MMPLL_D4_D2, CLK_TOP_MMPLL_D4, 1,
	       2, CLK_PARENT_TOPCKGEN),
	FACTOR(CLK_TOP_MMPLL_D4_D4, CLK_TOP_MMPLL_D4, 1, 4, CLK_PARENT_TOPCKGEN),
	FACTOR(CLK_TOP_MMPLL_D5, CLK_APMIXED_MMPLL, 1, 5, CLK_PARENT_APMIXED),
	FACTOR(CLK_TOP_MMPLL_D5_D2, CLK_TOP_MMPLL_D5, 1,
	       2, CLK_PARENT_TOPCKGEN),
	FACTOR(CLK_TOP_MMPLL_D5_D4, CLK_TOP_MMPLL_D5, 1,
	       4, CLK_PARENT_TOPCKGEN),
	FACTOR(CLK_TOP_MMPLL_D6, CLK_APMIXED_MMPLL, 1, 6, CLK_PARENT_APMIXED),
	FACTOR(CLK_TOP_MMPLL_D7, CLK_APMIXED_MMPLL, 1, 7, CLK_PARENT_APMIXED),
	FACTOR(CLK_TOP_MFGPLL_CK, CLK_APMIXED_MFGPLL, 1, 1, CLK_PARENT_APMIXED),
	FACTOR(CLK_TOP_MSDCPLL_CK, CLK_APMIXED_MSDCPLL, 1,
	       1, CLK_PARENT_APMIXED),
	FACTOR(CLK_TOP_MSDCPLL_D2, CLK_APMIXED_MSDCPLL, 1,
	       2, CLK_PARENT_APMIXED),
	FACTOR(CLK_TOP_MSDCPLL_D4, CLK_APMIXED_MSDCPLL, 1,
	       4, CLK_PARENT_APMIXED),
	FACTOR(CLK_TOP_MSDCPLL_D8, CLK_APMIXED_MSDCPLL, 1,
	       8, CLK_PARENT_APMIXED),
	FACTOR(CLK_TOP_MSDCPLL_D16, CLK_APMIXED_MSDCPLL, 1,
	       16, CLK_PARENT_APMIXED),
	FACTOR(CLK_TOP_AD_OSC_CK, CLK_TOP_ULPOSC, 1, 1, CLK_PARENT_TOPCKGEN),
	FACTOR(CLK_TOP_OSC_D2, CLK_TOP_ULPOSC, 1, 2, CLK_PARENT_TOPCKGEN),
	FACTOR(CLK_TOP_OSC_D4, CLK_TOP_ULPOSC, 1, 4, CLK_PARENT_TOPCKGEN),
	FACTOR(CLK_TOP_OSC_D8, CLK_TOP_ULPOSC, 1, 8, CLK_PARENT_TOPCKGEN),
	FACTOR(CLK_TOP_OSC_D16, CLK_TOP_ULPOSC, 1, 16, CLK_PARENT_TOPCKGEN),
	FACTOR(CLK_TOP_UNIVPLL, CLK_APMIXED_UNIV2PLL, 1, 2, CLK_PARENT_APMIXED),
	FACTOR(CLK_TOP_UNIVPLL_D3_D16, CLK_TOP_UNIVPLL_D3, 1,
	       16, CLK_PARENT_TOPCKGEN),
};

static const int axi_parents[] = {
	CLK_TOP_CLK26M,
	CLK_TOP_SYSPLL_D2_D4,
	CLK_TOP_SYSPLL_D7,
	CLK_TOP_OSC_D4
};

static const int mm_parents[] = {
	CLK_TOP_CLK26M,
	CLK_TOP_MMPLL_D7,
	CLK_TOP_SYSPLL_D3,
	CLK_TOP_UNIVPLL_D2_D2,
	CLK_TOP_SYSPLL_D2_D2,
	CLK_TOP_SYSPLL_D3_D2
};

static const int img_parents[] = {
	CLK_TOP_CLK26M,
	CLK_TOP_MMPLL_D6,
	CLK_TOP_UNIVPLL_D3,
	CLK_TOP_SYSPLL_D3,
	CLK_TOP_UNIVPLL_D2_D2,
	CLK_TOP_SYSPLL_D2_D2,
	CLK_TOP_UNIVPLL_D3_D2,
	CLK_TOP_SYSPLL_D3_D2
};

static const int cam_parents[] = {
	CLK_TOP_CLK26M,
	CLK_TOP_SYSPLL_D2,
	CLK_TOP_MMPLL_D6,
	CLK_TOP_SYSPLL_D3,
	CLK_TOP_MMPLL_D7,
	CLK_TOP_UNIVPLL_D3,
	CLK_TOP_UNIVPLL_D2_D2,
	CLK_TOP_SYSPLL_D2_D2,
	CLK_TOP_SYSPLL_D3_D2,
	CLK_TOP_UNIVPLL_D3_D2
};

static const int dsp_parents[] = {
	CLK_TOP_CLK26M,
	CLK_TOP_MMPLL_D6,
	CLK_TOP_MMPLL_D7,
	CLK_TOP_UNIVPLL_D3,
	CLK_TOP_SYSPLL_D3,
	CLK_TOP_UNIVPLL_D2_D2,
	CLK_TOP_SYSPLL_D2_D2,
	CLK_TOP_UNIVPLL_D3_D2,
	CLK_TOP_SYSPLL_D3_D2
};

static const int dsp1_parents[] = {
	CLK_TOP_CLK26M,
	CLK_TOP_MMPLL_D6,
	CLK_TOP_MMPLL_D7,
	CLK_TOP_UNIVPLL_D3,
	CLK_TOP_SYSPLL_D3,
	CLK_TOP_UNIVPLL_D2_D2,
	CLK_TOP_SYSPLL_D2_D2,
	CLK_TOP_UNIVPLL_D3_D2,
	CLK_TOP_SYSPLL_D3_D2
};

static const int dsp2_parents[] = {
	CLK_TOP_CLK26M,
	CLK_TOP_MMPLL_D6,
	CLK_TOP_MMPLL_D7,
	CLK_TOP_UNIVPLL_D3,
	CLK_TOP_SYSPLL_D3,
	CLK_TOP_UNIVPLL_D2_D2,
	CLK_TOP_SYSPLL_D2_D2,
	CLK_TOP_UNIVPLL_D3_D2,
	CLK_TOP_SYSPLL_D3_D2
};

static const int ipu_if_parents[] = {
	CLK_TOP_CLK26M,
	CLK_TOP_MMPLL_D6,
	CLK_TOP_MMPLL_D7,
	CLK_TOP_UNIVPLL_D3,
	CLK_TOP_SYSPLL_D3,
	CLK_TOP_UNIVPLL_D2_D2,
	CLK_TOP_SYSPLL_D2_D2,
	CLK_TOP_UNIVPLL_D3_D2,
	CLK_TOP_SYSPLL_D3_D2
};

static const int mfg_parents[] = {
	CLK_TOP_CLK26M,
	CLK_TOP_MFGPLL_CK,
	CLK_TOP_UNIVPLL_D3,
	CLK_TOP_SYSPLL_D3
};

static const int f52m_mfg_parents[] = {
	CLK_TOP_CLK26M,
	CLK_TOP_UNIVPLL_D3_D2,
	CLK_TOP_UNIVPLL_D3_D4,
	CLK_TOP_UNIVPLL_D3_D8
};

static const int camtg_parents[] = {
	CLK_TOP_CLK26M,
	CLK_TOP_UNIVP_192M_D8,
	CLK_TOP_UNIVPLL_D3_D8,
	CLK_TOP_UNIVP_192M_D4,
	CLK_TOP_UNIVPLL_D3_D16,
	CLK_TOP_F26M_CK_D2,
	CLK_TOP_UNIVP_192M_D16,
	CLK_TOP_UNIVP_192M_D32
};

static const int camtg2_parents[] = {
	CLK_TOP_CLK26M,
	CLK_TOP_UNIVP_192M_D8,
	CLK_TOP_UNIVPLL_D3_D8,
	CLK_TOP_UNIVP_192M_D4,
	CLK_TOP_UNIVPLL_D3_D16,
	CLK_TOP_F26M_CK_D2,
	CLK_TOP_UNIVP_192M_D16,
	CLK_TOP_UNIVP_192M_D32
};

static const int camtg3_parents[] = {
	CLK_TOP_CLK26M,
	CLK_TOP_UNIVP_192M_D8,
	CLK_TOP_UNIVPLL_D3_D8,
	CLK_TOP_UNIVP_192M_D4,
	CLK_TOP_UNIVPLL_D3_D16,
	CLK_TOP_F26M_CK_D2,
	CLK_TOP_UNIVP_192M_D16,
	CLK_TOP_UNIVP_192M_D32
};

static const int camtg4_parents[] = {
	CLK_TOP_CLK26M,
	CLK_TOP_UNIVP_192M_D8,
	CLK_TOP_UNIVPLL_D3_D8,
	CLK_TOP_UNIVP_192M_D4,
	CLK_TOP_UNIVPLL_D3_D16,
	CLK_TOP_F26M_CK_D2,
	CLK_TOP_UNIVP_192M_D16,
	CLK_TOP_UNIVP_192M_D32
};

static const int uart_parents[] = {
	CLK_TOP_CLK26M,
	CLK_TOP_UNIVPLL_D3_D8
};

static const int spi_parents[] = {
	CLK_TOP_CLK26M,
	CLK_TOP_SYSPLL_D5_D2,
	CLK_TOP_SYSPLL_D3_D4,
	CLK_TOP_MSDCPLL_D4
};

static const int msdc50_hclk_parents[] = {
	CLK_TOP_CLK26M,
	CLK_TOP_SYSPLL_D2_D2,
	CLK_TOP_SYSPLL_D3_D2
};

static const int msdc50_0_parents[] = {
	CLK_TOP_CLK26M,
	CLK_TOP_MSDCPLL_CK,
	CLK_TOP_MSDCPLL_D2,
	CLK_TOP_UNIVPLL_D2_D4,
	CLK_TOP_SYSPLL_D3_D2,
	CLK_TOP_UNIVPLL_D2_D2
};

static const int msdc30_1_parents[] = {
	CLK_TOP_CLK26M,
	CLK_TOP_UNIVPLL_D3_D2,
	CLK_TOP_SYSPLL_D3_D2,
	CLK_TOP_SYSPLL_D7,
	CLK_TOP_MSDCPLL_D2
};

static const int msdc30_2_parents[] = {
	CLK_TOP_CLK26M,
	CLK_TOP_UNIVPLL_D3_D2,
	CLK_TOP_SYSPLL_D3_D2,
	CLK_TOP_SYSPLL_D7,
	CLK_TOP_MSDCPLL_D2
};

static const int audio_parents[] = {
	CLK_TOP_CLK26M,
	CLK_TOP_SYSPLL_D5_D4,
	CLK_TOP_SYSPLL_D7_D4,
	CLK_TOP_SYSPLL_D2_D16
};

static const int aud_intbus_parents[] = {
	CLK_TOP_CLK26M,
	CLK_TOP_SYSPLL_D2_D4,
	CLK_TOP_SYSPLL_D7_D2
};

static const int pmicspi_parents[] = {
	CLK_TOP_CLK26M,
	CLK_TOP_SYSPLL_D2_D8,
	CLK_TOP_OSC_D8
};

static const int fpwrap_ulposc_parents[] = {
	CLK_TOP_CLK26M,
	CLK_TOP_OSC_D16,
	CLK_TOP_OSC_D4,
	CLK_TOP_OSC_D8
};

static const int atb_parents[] = {
	CLK_TOP_CLK26M,
	CLK_TOP_SYSPLL_D2_D2,
	CLK_TOP_SYSPLL_D5
};

static const int sspm_parents[] = {
	CLK_TOP_CLK26M,
	CLK_TOP_UNIVPLL_D2_D4,
	CLK_TOP_SYSPLL_D2_D2,
	CLK_TOP_UNIVPLL_D2_D2,
	CLK_TOP_SYSPLL_D3
};

static const int dpi0_parents[] = {
	CLK_TOP_CLK26M,
	CLK_TOP_TVDPLL_D2,
	CLK_TOP_TVDPLL_D4,
	CLK_TOP_TVDPLL_D8,
	CLK_TOP_TVDPLL_D16,
	CLK_TOP_UNIVPLL_D5_D2,
	CLK_TOP_UNIVPLL_D3_D4,
	CLK_TOP_SYSPLL_D3_D4,
	CLK_TOP_UNIVPLL_D3_D8
};

static const int scam_parents[] = {
	CLK_TOP_CLK26M,
	CLK_TOP_SYSPLL_D5_D2
};

static const int disppwm_parents[] = {
	CLK_TOP_CLK26M,
	CLK_TOP_UNIVPLL_D3_D4,
	CLK_TOP_OSC_D2,
	CLK_TOP_OSC_D4,
	CLK_TOP_OSC_D16
};

static const int usb_top_parents[] = {
	CLK_TOP_CLK26M,
	CLK_TOP_UNIVPLL_D5_D4,
	CLK_TOP_UNIVPLL_D3_D4,
	CLK_TOP_UNIVPLL_D5_D2
};

static const int ssusb_top_xhci_parents[] = {
	CLK_TOP_CLK26M,
	CLK_TOP_UNIVPLL_D5_D4,
	CLK_TOP_UNIVPLL_D3_D4,
	CLK_TOP_UNIVPLL_D5_D2
};

static const int spm_parents[] = {
	CLK_TOP_CLK26M,
	CLK_TOP_SYSPLL_D2_D8
};

static const int i2c_parents[] = {
	CLK_TOP_CLK26M,
	CLK_TOP_SYSPLL_D2_D8,
	CLK_TOP_UNIVPLL_D5_D2
};

static const int scp_parents[] = {
	CLK_TOP_CLK26M,
	CLK_TOP_UNIVPLL_D2_D8,
	CLK_TOP_SYSPLL_D5,
	CLK_TOP_SYSPLL_D2_D2,
	CLK_TOP_UNIVPLL_D2_D2,
	CLK_TOP_SYSPLL_D3,
	CLK_TOP_UNIVPLL_D3
};

static const int seninf_parents[] = {
	CLK_TOP_CLK26M,
	CLK_TOP_UNIVPLL_D2_D2,
	CLK_TOP_UNIVPLL_D3_D2,
	CLK_TOP_UNIVPLL_D2_D4
};

static const int dxcc_parents[] = {
	CLK_TOP_CLK26M,
	CLK_TOP_SYSPLL_D2_D2,
	CLK_TOP_SYSPLL_D2_D4,
	CLK_TOP_SYSPLL_D2_D8
};

static const int aud_engen1_parents[] = {
	CLK_TOP_CLK26M,
	CLK_TOP_APLL1_D2,
	CLK_TOP_APLL1_D4,
	CLK_TOP_APLL1_D8
};

static const int aud_engen2_parents[] = {
	CLK_TOP_CLK26M,
	CLK_TOP_APLL2_D2,
	CLK_TOP_APLL2_D4,
	CLK_TOP_APLL2_D8
};

static const int faes_ufsfde_parents[] = {
	CLK_TOP_CLK26M,
	CLK_TOP_SYSPLL_D2,
	CLK_TOP_SYSPLL_D2_D2,
	CLK_TOP_SYSPLL_D3,
	CLK_TOP_SYSPLL_D2_D4,
	CLK_TOP_UNIVPLL_D3
};

static const int fufs_parents[] = {
	CLK_TOP_CLK26M,
	CLK_TOP_SYSPLL_D2_D4,
	CLK_TOP_SYSPLL_D2_D8,
	CLK_TOP_SYSPLL_D2_D16
};

static const int aud_1_parents[] = {
	CLK_TOP_CLK26M,
	CLK_TOP_APLL1_CK
};

static const int aud_2_parents[] = {
	CLK_TOP_CLK26M,
	CLK_TOP_APLL2_CK
};

static const struct mtk_composite top_muxes[] = {
	/* CLK_CFG_0 */
	MUX(CLK_TOP_MUX_AXI, axi_parents, 0x40, 0, 2),
	MUX(CLK_TOP_MUX_MM, mm_parents, 0x40, 8, 3),
	MUX(CLK_TOP_MUX_IMG, img_parents, 0x40, 16, 3),
	MUX(CLK_TOP_MUX_CAM, cam_parents, 0x40, 24, 4),
	/* CLK_CFG_1 */
	MUX(CLK_TOP_MUX_DSP, dsp_parents, 0x50, 0, 4),
	MUX(CLK_TOP_MUX_DSP1, dsp1_parents, 0x50, 8, 4),
	MUX(CLK_TOP_MUX_DSP2, dsp2_parents, 0x50, 16, 4),
	MUX(CLK_TOP_MUX_IPU_IF, ipu_if_parents, 0x50, 24, 4),
	/* CLK_CFG_2 */
	MUX(CLK_TOP_MUX_MFG, mfg_parents, 0x60, 0, 2),
	MUX(CLK_TOP_MUX_F52M_MFG, f52m_mfg_parents, 0x60, 8, 2),
	MUX(CLK_TOP_MUX_CAMTG, camtg_parents, 0x60, 16, 3),
	MUX(CLK_TOP_MUX_CAMTG2, camtg2_parents, 0x60, 24, 3),
	/* CLK_CFG_3 */
	MUX(CLK_TOP_MUX_CAMTG3, camtg3_parents, 0x70, 0, 3),
	MUX(CLK_TOP_MUX_CAMTG4, camtg4_parents, 0x70, 8, 3),
	MUX(CLK_TOP_MUX_UART, uart_parents, 0x70, 16, 1),
	MUX(CLK_TOP_MUX_SPI, spi_parents, 0x70, 24, 2),
	/* CLK_CFG_4 */
	MUX(CLK_TOP_MUX_MSDC50_0_HCLK, msdc50_hclk_parents, 0x80, 0, 2),
	MUX(CLK_TOP_MUX_MSDC50_0, msdc50_0_parents, 0x80, 8, 3),
	MUX(CLK_TOP_MUX_MSDC30_1, msdc30_1_parents, 0x80, 16, 3),
	MUX(CLK_TOP_MUX_MSDC30_2, msdc30_2_parents, 0x80, 24, 3),
	/* CLK_CFG_5 */
	MUX(CLK_TOP_MUX_AUDIO, audio_parents, 0x90, 0, 2),
	MUX(CLK_TOP_MUX_AUD_INTBUS, aud_intbus_parents, 0x90, 8, 2),
	MUX(CLK_TOP_MUX_PMICSPI, pmicspi_parents, 0x90, 16, 2),
	MUX(CLK_TOP_MUX_FPWRAP_ULPOSC, fpwrap_ulposc_parents, 0x90, 24, 2),
	/* CLK_CFG_6 */
	MUX(CLK_TOP_MUX_ATB, atb_parents, 0xa0, 0, 2),
	MUX(CLK_TOP_MUX_SSPM, sspm_parents, 0xa0, 8, 3),
	MUX(CLK_TOP_MUX_DPI0, dpi0_parents, 0xa0, 16, 4),
	MUX(CLK_TOP_MUX_SCAM, scam_parents, 0xa0, 24, 1),
	/* CLK_CFG_7 */
	MUX(CLK_TOP_MUX_DISP_PWM, disppwm_parents, 0xb0, 0, 3),
	MUX(CLK_TOP_MUX_USB_TOP, usb_top_parents, 0xb0, 8, 2),
	MUX(CLK_TOP_MUX_SSUSB_TOP_XHCI, ssusb_top_xhci_parents, 0xb0, 16, 2),
	MUX(CLK_TOP_MUX_SPM, spm_parents, 0xb0, 24, 1),
	/* CLK_CFG_8 */
	MUX(CLK_TOP_MUX_I2C, i2c_parents, 0xc0, 0, 2),
	MUX(CLK_TOP_MUX_SCP, scp_parents, 0xc0, 8, 3),
	MUX(CLK_TOP_MUX_SENINF, seninf_parents, 0xc0, 16, 2),
	MUX(CLK_TOP_MUX_DXCC, dxcc_parents, 0xc0, 24, 2),
	/* CLK_CFG_9 */
	MUX(CLK_TOP_MUX_AUD_ENG1, aud_engen1_parents, 0xd0, 0, 2),
	MUX(CLK_TOP_MUX_AUD_ENG2, aud_engen2_parents, 0xd0, 8, 2),
	MUX(CLK_TOP_MUX_FAES_UFSFDE, faes_ufsfde_parents, 0xd0, 16, 3),
	MUX(CLK_TOP_MUX_FUFS, fufs_parents, 0xd0, 24, 2),
	/* CLK_CFG_10 */
	MUX(CLK_TOP_MUX_AUD_1, aud_1_parents, 0xe0, 0, 1),
	MUX(CLK_TOP_MUX_AUD_2, aud_2_parents, 0xe0, 8, 1),
};

static const struct mtk_clk_tree mt8183_clk_tree = {
	.xtal_rate = 26 * MHZ,
	.xtal2_rate = 26 * MHZ,
	.fdivs_offs = CLK_TOP_CLK13M,
	.muxes_offs = CLK_TOP_MUX_AXI,
	.plls = apmixed_plls,
	.fclks = top_fixed_clks,
	.fdivs = top_fixed_divs,
	.muxes = top_muxes,
};

static const struct mtk_gate_regs infra0_cg_regs = {
	.set_ofs = 0x80,
	.clr_ofs = 0x84,
	.sta_ofs = 0x90,
};

static const struct mtk_gate_regs infra1_cg_regs = {
	.set_ofs = 0x88,
	.clr_ofs = 0x8c,
	.sta_ofs = 0x94,
};

static const struct mtk_gate_regs infra2_cg_regs = {
	.set_ofs = 0xa4,
	.clr_ofs = 0xa8,
	.sta_ofs = 0xac,
};

static const struct mtk_gate_regs infra3_cg_regs = {
	.set_ofs = 0xc0,
	.clr_ofs = 0xc4,
	.sta_ofs = 0xc8,
};

#define GATE_INFRA0(_id, _parent, _shift) {		\
		.id = _id,						\
		.parent = _parent,					\
		.regs = &infra0_cg_regs,				\
		.shift = _shift,					\
		.flags = CLK_GATE_SETCLR | CLK_PARENT_TOPCKGEN,		\
	}

#define GATE_INFRA1(_id, _parent, _shift) {		\
		.id = _id,						\
		.parent = _parent,					\
		.regs = &infra1_cg_regs,				\
		.shift = _shift,					\
		.flags = CLK_GATE_SETCLR | CLK_PARENT_TOPCKGEN,		\
	}

#define GATE_INFRA2(_id, _parent, _shift) {		\
		.id = _id,						\
		.parent = _parent,					\
		.regs = &infra2_cg_regs,				\
		.shift = _shift,					\
		.flags = CLK_GATE_SETCLR | CLK_PARENT_TOPCKGEN,		\
	}

#define GATE_INFRA3(_id, _parent, _shift) {		\
		.id = _id,						\
		.parent = _parent,					\
		.regs = &infra3_cg_regs,				\
		.shift = _shift,					\
		.flags = CLK_GATE_SETCLR | CLK_PARENT_TOPCKGEN,		\
	}

static const struct mtk_gate infra_clks[] = {
	/* INFRA0 */
	GATE_INFRA0(CLK_INFRA_PMIC_TMR, CLK_TOP_MUX_AXI, 0),
	GATE_INFRA0(CLK_INFRA_PMIC_AP, CLK_TOP_MUX_AXI, 1),
	GATE_INFRA0(CLK_INFRA_PMIC_MD, CLK_TOP_MUX_AXI, 2),
	GATE_INFRA0(CLK_INFRA_PMIC_CONN, CLK_TOP_MUX_AXI, 3),
	GATE_INFRA0(CLK_INFRA_SCPSYS, CLK_TOP_MUX_SCP, 4),
	GATE_INFRA0(CLK_INFRA_SEJ, CLK_TOP_CLK26M, 5),
	GATE_INFRA0(CLK_INFRA_APXGPT, CLK_TOP_MUX_AXI, 6),
	GATE_INFRA0(CLK_INFRA_ICUSB, CLK_TOP_MUX_AXI, 8),
	GATE_INFRA0(CLK_INFRA_GCE, CLK_TOP_MUX_AXI, 9),
	GATE_INFRA0(CLK_INFRA_THERM, CLK_TOP_MUX_AXI, 10),
	GATE_INFRA0(CLK_INFRA_I2C0, CLK_TOP_MUX_I2C, 11),
	GATE_INFRA0(CLK_INFRA_I2C1, CLK_TOP_MUX_I2C, 12),
	GATE_INFRA0(CLK_INFRA_I2C2, CLK_TOP_MUX_I2C, 13),
	GATE_INFRA0(CLK_INFRA_I2C3, CLK_TOP_MUX_I2C, 14),
	GATE_INFRA0(CLK_INFRA_PWM_HCLK, CLK_TOP_MUX_AXI, 15),
	GATE_INFRA0(CLK_INFRA_PWM1, CLK_TOP_MUX_I2C, 16),
	GATE_INFRA0(CLK_INFRA_PWM2, CLK_TOP_MUX_I2C, 17),
	GATE_INFRA0(CLK_INFRA_PWM3, CLK_TOP_MUX_I2C, 18),
	GATE_INFRA0(CLK_INFRA_PWM4, CLK_TOP_MUX_I2C, 19),
	GATE_INFRA0(CLK_INFRA_PWM, CLK_TOP_MUX_I2C, 21),
	GATE_INFRA0(CLK_INFRA_UART0, CLK_TOP_MUX_UART, 22),
	GATE_INFRA0(CLK_INFRA_UART1, CLK_TOP_MUX_UART, 23),
	GATE_INFRA0(CLK_INFRA_UART2, CLK_TOP_MUX_UART, 24),
	GATE_INFRA0(CLK_INFRA_UART3, CLK_TOP_MUX_UART, 25),
	GATE_INFRA0(CLK_INFRA_GCE_26M, CLK_TOP_MUX_AXI, 27),
	GATE_INFRA0(CLK_INFRA_CQ_DMA_FPC, CLK_TOP_MUX_AXI, 28),
	GATE_INFRA0(CLK_INFRA_BTIF, CLK_TOP_MUX_AXI, 31),
	/* INFRA1 */
	GATE_INFRA1(CLK_INFRA_SPI0, CLK_TOP_MUX_SPI, 1),
	GATE_INFRA1(CLK_INFRA_MSDC0, CLK_TOP_MUX_MSDC50_0_HCLK, 2),
	GATE_INFRA1(CLK_INFRA_MSDC1, CLK_TOP_MUX_AXI, 4),
	GATE_INFRA1(CLK_INFRA_MSDC2, CLK_TOP_MUX_AXI, 5),
	GATE_INFRA1(CLK_INFRA_MSDC0_SCK, CLK_TOP_MUX_MSDC50_0, 6),
	GATE_INFRA1(CLK_INFRA_DVFSRC, CLK_TOP_CLK26M, 7),
	GATE_INFRA1(CLK_INFRA_GCPU, CLK_TOP_MUX_AXI, 8),
	GATE_INFRA1(CLK_INFRA_TRNG, CLK_TOP_MUX_AXI, 9),
	GATE_INFRA1(CLK_INFRA_AUXADC, CLK_TOP_CLK26M, 10),
	GATE_INFRA1(CLK_INFRA_CPUM, CLK_TOP_MUX_AXI, 11),
	GATE_INFRA1(CLK_INFRA_CCIF1_AP, CLK_TOP_MUX_AXI, 12),
	GATE_INFRA1(CLK_INFRA_CCIF1_MD, CLK_TOP_MUX_AXI, 13),
	GATE_INFRA1(CLK_INFRA_AUXADC_MD, CLK_TOP_CLK26M, 14),
	GATE_INFRA1(CLK_INFRA_MSDC1_SCK, CLK_TOP_MUX_MSDC30_1, 16),
	GATE_INFRA1(CLK_INFRA_MSDC2_SCK, CLK_TOP_MUX_MSDC30_2, 17),
	GATE_INFRA1(CLK_INFRA_AP_DMA, CLK_TOP_MUX_AXI, 18),
	GATE_INFRA1(CLK_INFRA_XIU, CLK_TOP_MUX_AXI, 19),
	GATE_INFRA1(CLK_INFRA_DEVICE_APC, CLK_TOP_MUX_AXI, 20),
	GATE_INFRA1(CLK_INFRA_CCIF_AP, CLK_TOP_MUX_AXI, 23),
	GATE_INFRA1(CLK_INFRA_DEBUGSYS, CLK_TOP_MUX_AXI, 24),
	GATE_INFRA1(CLK_INFRA_AUDIO, CLK_TOP_MUX_AXI, 25),
	GATE_INFRA1(CLK_INFRA_CCIF_MD, CLK_TOP_MUX_AXI, 26),
	GATE_INFRA1(CLK_INFRA_DXCC_SEC_CORE, CLK_TOP_MUX_DXCC, 27),
	GATE_INFRA1(CLK_INFRA_DXCC_AO, CLK_TOP_MUX_DXCC, 28),
	GATE_INFRA1(CLK_INFRA_DEVMPU_BCLK, CLK_TOP_MUX_AXI, 30),
	GATE_INFRA1(CLK_INFRA_DRAMC_F26M, CLK_TOP_CLK26M, 31),
	/* INFRA2 */
	GATE_INFRA2(CLK_INFRA_IRTX, CLK_TOP_CLK26M, 0),
	GATE_INFRA2(CLK_INFRA_USB, CLK_TOP_MUX_USB_TOP, 1),
	GATE_INFRA2(CLK_INFRA_DISP_PWM, CLK_TOP_MUX_AXI, 2),
	GATE_INFRA2(CLK_INFRA_CLDMA_BCLK, CLK_TOP_MUX_AXI, 3),
	GATE_INFRA2(CLK_INFRA_AUDIO_26M_BCLK, CLK_TOP_CLK26M, 4),
	GATE_INFRA2(CLK_INFRA_SPI1, CLK_TOP_MUX_SPI, 6),
	GATE_INFRA2(CLK_INFRA_I2C4, CLK_TOP_MUX_I2C, 7),
	GATE_INFRA2(CLK_INFRA_MODEM_TEMP_SHARE, CLK_TOP_CLK26M, 8),
	GATE_INFRA2(CLK_INFRA_SPI2, CLK_TOP_MUX_SPI, 9),
	GATE_INFRA2(CLK_INFRA_SPI3, CLK_TOP_MUX_SPI, 10),
	GATE_INFRA2(CLK_INFRA_UNIPRO_SCK, CLK_TOP_MUX_SSUSB_TOP_XHCI, 11),
	GATE_INFRA2(CLK_INFRA_UNIPRO_TICK, CLK_TOP_MUX_FUFS, 12),
	GATE_INFRA2(CLK_INFRA_UFS_MP_SAP_BCLK, CLK_TOP_MUX_FUFS, 13),
	GATE_INFRA2(CLK_INFRA_MD32_BCLK, CLK_TOP_MUX_AXI, 14),
	GATE_INFRA2(CLK_INFRA_UNIPRO_MBIST, CLK_TOP_MUX_AXI, 16),
	GATE_INFRA2(CLK_INFRA_I2C5, CLK_TOP_MUX_I2C, 18),
	GATE_INFRA2(CLK_INFRA_I2C5_ARBITER, CLK_TOP_MUX_I2C, 19),
	GATE_INFRA2(CLK_INFRA_I2C5_IMM, CLK_TOP_MUX_I2C, 20),
	GATE_INFRA2(CLK_INFRA_I2C1_ARBITER, CLK_TOP_MUX_I2C, 21),
	GATE_INFRA2(CLK_INFRA_I2C1_IMM, CLK_TOP_MUX_I2C, 22),
	GATE_INFRA2(CLK_INFRA_I2C2_ARBITER, CLK_TOP_MUX_I2C, 23),
	GATE_INFRA2(CLK_INFRA_I2C2_IMM, CLK_TOP_MUX_I2C, 24),
	GATE_INFRA2(CLK_INFRA_SPI4, CLK_TOP_MUX_SPI, 25),
	GATE_INFRA2(CLK_INFRA_SPI5, CLK_TOP_MUX_SPI, 26),
	GATE_INFRA2(CLK_INFRA_CQ_DMA, CLK_TOP_MUX_AXI, 27),
	GATE_INFRA2(CLK_INFRA_UFS, CLK_TOP_MUX_FUFS, 28),
	GATE_INFRA2(CLK_INFRA_AES_UFSFDE, CLK_TOP_MUX_FAES_UFSFDE, 29),
	GATE_INFRA2(CLK_INFRA_UFS_TICK, CLK_TOP_MUX_FUFS, 30),
	/* INFRA3 */
	GATE_INFRA3(CLK_INFRA_MSDC0_SELF, CLK_TOP_MUX_MSDC50_0, 0),
	GATE_INFRA3(CLK_INFRA_MSDC1_SELF, CLK_TOP_MUX_MSDC50_0, 1),
	GATE_INFRA3(CLK_INFRA_MSDC2_SELF, CLK_TOP_MUX_MSDC50_0, 2),
	GATE_INFRA3(CLK_INFRA_UFS_AXI, CLK_TOP_MUX_AXI, 5),
	GATE_INFRA3(CLK_INFRA_I2C6, CLK_TOP_MUX_I2C, 6),
	GATE_INFRA3(CLK_INFRA_AP_MSDC0, CLK_TOP_MUX_MSDC50_0_HCLK, 7),
	GATE_INFRA3(CLK_INFRA_MD_MSDC0, CLK_TOP_MUX_MSDC50_0_HCLK, 8),
	GATE_INFRA3(CLK_INFRA_CCIF2_AP, CLK_TOP_MUX_AXI, 16),
	GATE_INFRA3(CLK_INFRA_CCIF2_MD, CLK_TOP_MUX_AXI, 17),
	GATE_INFRA3(CLK_INFRA_CCIF3_AP, CLK_TOP_MUX_AXI, 18),
	GATE_INFRA3(CLK_INFRA_CCIF3_MD, CLK_TOP_MUX_AXI, 19),
	GATE_INFRA3(CLK_INFRA_SEJ_F13M, CLK_TOP_CLK26M, 20),
	GATE_INFRA3(CLK_INFRA_AES_BCLK, CLK_TOP_MUX_AXI, 21),
	GATE_INFRA3(CLK_INFRA_I2C7, CLK_TOP_MUX_I2C, 22),
	GATE_INFRA3(CLK_INFRA_I2C8, CLK_TOP_MUX_I2C, 23),
	GATE_INFRA3(CLK_INFRA_FBIST2FPC, CLK_TOP_MUX_MSDC50_0, 24),
};

static int mt8183_apmixedsys_probe(struct udevice *dev)
{
	return mtk_common_clk_init(dev, &mt8183_clk_tree);
}

static int mt8183_topckgen_probe(struct udevice *dev)
{
	return mtk_common_clk_init(dev, &mt8183_clk_tree);
}

static int mt8183_infracfg_probe(struct udevice *dev)
{
	return mtk_common_clk_gate_init(dev, &mt8183_clk_tree, infra_clks);
}

static const struct udevice_id mt8183_apmixed_compat[] = {
	{ .compatible = "mediatek,mt8183-apmixedsys", },
	{ }
};

static const struct udevice_id mt8183_topckgen_compat[] = {
	{ .compatible = "mediatek,mt8183-topckgen", },
	{ }
};

static const struct udevice_id mt8183_infracfg_compat[] = {
	{ .compatible = "mediatek,mt8183-infracfg", },
	{ }
};

U_BOOT_DRIVER(mtk_clk_apmixedsys) = {
	.name = "mt8183-apmixedsys",
	.id = UCLASS_CLK,
	.of_match = mt8183_apmixed_compat,
	.probe = mt8183_apmixedsys_probe,
	.priv_auto = sizeof(struct mtk_clk_priv),
	.ops = &mtk_clk_apmixedsys_ops,
	.flags = DM_FLAG_PRE_RELOC,
};

U_BOOT_DRIVER(mtk_clk_topckgen) = {
	.name = "mt8183-topckgen",
	.id = UCLASS_CLK,
	.of_match = mt8183_topckgen_compat,
	.probe = mt8183_topckgen_probe,
	.priv_auto = sizeof(struct mtk_clk_priv),
	.ops = &mtk_clk_topckgen_ops,
	.flags = DM_FLAG_PRE_RELOC,
};

U_BOOT_DRIVER(mtk_clk_infracfg) = {
	.name = "mt8183-infracfg",
	.id = UCLASS_CLK,
	.of_match = mt8183_infracfg_compat,
	.probe = mt8183_infracfg_probe,
	.priv_auto = sizeof(struct mtk_clk_priv),
	.ops = &mtk_clk_gate_ops,
	.flags = DM_FLAG_PRE_RELOC,
};
