/**
 *    Copyright (C) 2018-present MongoDB, Inc.
 *
 *    This program is free software: you can redistribute it and/or modify
 *    it under the terms of the Server Side Public License, version 1,
 *    as published by MongoDB, Inc.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    Server Side Public License for more details.
 *
 *    You should have received a copy of the Server Side Public License
 *    along with this program. If not, see
 *    <http://www.mongodb.com/licensing/server-side-public-license>.
 *
 *    As a special exception, the copyright holders give permission to link the
 *    code of portions of this program with the OpenSSL library under certain
 *    conditions as described in each individual source file and distribute
 *    linked combinations including the program with the OpenSSL library. You
 *    must comply with the Server Side Public License in all respects for
 *    all of the code used other than as permitted herein. If you modify file(s)
 *    with this exception, you may extend this exception to your version of the
 *    file(s), but you are not obligated to do so. If you do not wish to do so,
 *    delete this exception statement from your version. If you delete this
 *    exception statement from all source files in the program, then also delete
 *    it in the license file.
 */

#include "mongo/platform/basic.h"

#include "mongo/db/ftdc/ftdc_mongod.h"

#include <boost/filesystem.hpp>

#include "mongo/bson/bsonobjbuilder.h"
#include "mongo/db/ftdc/constants.h"
#include "mongo/db/ftdc/controller.h"
#include "mongo/db/ftdc/ftdc_mongod_gen.h"
#include "mongo/db/ftdc/ftdc_server.h"
#include "mongo/db/namespace_string.h"
#include "mongo/db/repl/replication_coordinator.h"
#include "mongo/db/storage/storage_options.h"
#include "mongo/transport/transport_layer_ftdc_collector.h"
#include "mongo/util/assert_util.h"

namespace mongo {

Status validateCollectionStatsNamespaces(const std::vector<std::string> value,
                                         const boost::optional<TenantId>& tenantId) {
    try {
        for (const auto& nsStr : value) {
            NamespaceString ns(nsStr);

            if (!ns.isValid()) {
                return Status(ErrorCodes::BadValue,
                              fmt::format("'{}' is not a valid namespace", nsStr));
            }
        }
    } catch (...) {
        return exceptionToStatus();
    }

    return Status::OK();
}

namespace {

class FTDCCollectionStatsCollector final : public FTDCCollectorInterface {
public:
    bool hasData() const override {
        return !gDiagnosticDataCollectionStatsNamespaces->empty();
    }

    void collect(OperationContext* opCtx, BSONObjBuilder& builder) override {
        std::vector<std::string> namespaces = gDiagnosticDataCollectionStatsNamespaces.get();

        for (const auto& nsStr : namespaces) {

            try {
                NamespaceString ns(nsStr);
                auto result = CommandHelpers::runCommandDirectly(
                    opCtx,
                    OpMsgRequest::fromDBAndBody(
                        ns.db(),
                        BSON("aggregate" << ns.coll() << "cursor" << BSONObj{} << "pipeline"
                                         << BSON_ARRAY(BSON("$collStats" << BSON(
                                                                "storageStats" << BSON(
                                                                    "waitForLock" << false)))))));
                builder.append(nsStr, result["cursor"]["firstBatch"]["0"].Obj());

            } catch (...) {
                Status s = exceptionToStatus();
                builder.append("error", s.toString());
            }
        }
    }

    std::string name() const override {
        return "collectionStats";
    }
};

static const BSONArray pipelineObj =
    BSONArrayBuilder{}
        .append(BSONObjBuilder{}
                    .append("$collStats",
                            BSONObjBuilder{}
                                .append("storageStats",
                                        BSONObjBuilder{}
                                            .append("waitForLock", false)
                                            .append("numericOnly", true)
                                            .obj())
                                .obj())
                    .obj())
        .arr();

static const BSONObj replSetGetStatusObj =
    BSONObjBuilder{}.append("replSetGetStatus", 1).append("initialSync", 0).obj();

static const BSONObj getDefaultRWConcernObj =
    BSONObjBuilder{}.append("getDefaultRWConcern", 1).append("inMemory", true).obj();

repl::ReplicationCoordinator* getGlobalRC() {
    return repl::ReplicationCoordinator::get(getGlobalServiceContext());
}

bool isRepl(const repl::ReplicationCoordinator& rc) {
    return rc.getReplicationMode() != repl::ReplicationCoordinator::modeNone;
}

bool isArbiter(const repl::ReplicationCoordinator& rc) {
    return isRepl(rc) && rc.getMemberState().arbiter();
}

bool isDataStoringNode() {
    auto rc = getGlobalRC();
    return !(rc && isArbiter(*rc));
}

std::unique_ptr<FTDCCollectorInterface> makeFilteredCollector(
    std::function<bool()> pred, std::unique_ptr<FTDCCollectorInterface> collector) {
    return std::make_unique<FilteredFTDCCollector>(std::move(pred), std::move(collector));
}

void registerMongoDCollectors(FTDCController* controller) {

    // These metrics are only collected if replication is enabled
    if (auto rc = getGlobalRC(); rc && isRepl(*rc)) {
        // CmdReplSetGetStatus
        controller->addPeriodicCollector(std::make_unique<FTDCSimpleInternalCommandCollector>(
            "replSetGetStatus", "replSetGetStatus", "", replSetGetStatusObj));

        // CollectionStats
        controller->addPeriodicCollector(
            makeFilteredCollector(isDataStoringNode,
                                  std::make_unique<FTDCSimpleInternalCommandCollector>(
                                      "aggregate",
                                      "local.oplog.rs.stats",
                                      "local",
                                      BSONObjBuilder{}
                                          .append("aggregate", "oplog.rs")
                                          .append("cursor", BSONObj{})
                                          .append("pipeline", pipelineObj)
                                          .obj())));

        if (!serverGlobalParams.clusterRole.exclusivelyHasShardRole()) {
            // GetDefaultRWConcern
            controller->addOnRotateCollector(std::make_unique<FTDCSimpleInternalCommandCollector>(
                "getDefaultRWConcern", "getDefaultRWConcern", "", getDefaultRWConcernObj));
        }
    }

    controller->addPeriodicCollector(
        makeFilteredCollector(isDataStoringNode, std::make_unique<FTDCCollectionStatsCollector>()));

    controller->addPeriodicCollector(std::make_unique<transport::TransportLayerFTDCCollector>());
}

}  // namespace

void startMongoDFTDC() {
    auto dir = getFTDCDirectoryPathParameter();

    if (dir.empty()) {
        dir = storageGlobalParams.dbpath;
        dir /= kFTDCDefaultDirectory.toString();
    }

    startFTDC(dir, FTDCStartMode::kStart, registerMongoDCollectors);
}

void stopMongoDFTDC() {
    stopFTDC();
}

}  // namespace mongo
