/*
 * Copyright (c) 1996, 1997, 1999, 2000, 2004, 2009, 2010, 2012, 2013, 2016, 2019, 2023
 * The Regents of the University of California. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University nor the names of its contributors
 *       may be used to endorse or promote products derived from this software
 *       without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#ifndef lint
static const char rcsid[] __attribute__((unused)) =
    "@(#) $Id: util.c 1537 2023-09-05 17:50:56Z leres $ (LBL)";
#endif

/*
 * util - arpwatch utility routines
 */

#include <sys/types.h>
#include <sys/file.h>

#include <errno.h>
#include <fcntl.h>
#ifdef HAVE_MEMORY_H
#include <memory.h>
#endif
#include <netdb.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>

#include "gnuc.h"
#ifdef HAVE_OS_PROTO_H
#include "os-proto.h"
#endif

#include "arpwatch.h"
#include "db.h"
#include "ec.h"
#include "file.h"
#include "util.h"

char *arpdir = ARPDIR;
char *arpfile = ARPFILE;
char *ethercodes = ETHERCODES;

/* Broadcast ethernet addresses */
u_char zero[6] = { 0, 0, 0, 0, 0, 0 };
u_char allones[6] = { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff };

/* VRRP/CARP ethernet prefix */
u_char vrrp_prefix[5] = { 0x00, 0x00, 0x5e, 0x00, 0x01 };

int debug;
int vrrpflag;
int zeroflag;
int zeropad = ZEROPAD;
int initializing = 1;			/* true if initializing */

/* syslog() helper routine */
void
dosyslog(int p, const char *s, u_int32_t a, const u_char *ea, const u_char *ha)
{
	char xbuf[64];

	/* No report until we're initialized */
	if (initializing)
		return;

	/* Display both ethernet addresses if they don't match */
	(void)strncpy(xbuf, e2str(ea), sizeof(xbuf));
	xbuf[sizeof(xbuf) - 1] = '\0';
	if (ha != NULL && memcmp(ea, ha, 6) != 0) {
		(void)strcat(xbuf, " (");
		(void)strcat(xbuf, e2str(ha));
		(void)strcat(xbuf, ")");
	}

	if (debug)
		fprintf(stderr, "%s: %s %s %s\n", prog, s, intoa(a), xbuf);
	else
		lg(p, "%s %s %s", s, intoa(a), xbuf);
}

static FILE *dumpf;

void
dumpone(u_int32_t a, const u_char *e, time_t t, const char *h)
{
	(void)fprintf(dumpf, "%s\t%s", e2str(e), intoa(a));
	if (t != 0 || h != NULL)
		(void)fprintf(dumpf, "\t%u", (u_int32_t)t);
	if (h != NULL && *h != '\0')
		(void)fprintf(dumpf, "\t%s", h);
	(void)putc('\n', dumpf);
}

int
dump(void)
{
	int fd;
	char oldarpfile[256], newarpfile[256];

	(void)snprintf(oldarpfile, sizeof(oldarpfile), "%s-", arpfile);
	(void)snprintf(newarpfile, sizeof(oldarpfile), "%s.new", arpfile);

	if ((fd = creat(newarpfile, 0644)) < 0) {
		lg(LOG_ERR, "creat(%s): %s", newarpfile, strerror(errno));
		return(0);
	}
	if ((dumpf = fdopen(fd, "w")) == NULL) {
		lg(LOG_ERR, "fdopen(%s): %s", newarpfile, strerror(errno));
		return(0);
	}

	(void)ent_loop(dumpone);
	if (ferror(dumpf)) {
		lg(LOG_ERR, "ferror %s: %s", newarpfile, strerror(errno));
		(void)remove(newarpfile);
		return(0);
	}

	if (fflush(dumpf) == EOF) {
		lg(LOG_ERR, "fflush %s: %s", newarpfile, strerror(errno));
		(void)remove(newarpfile);
		return(0);
	}
	if (fclose(dumpf) == EOF) {
		lg(LOG_ERR, "fclose %s: %s", newarpfile, strerror(errno));
		(void)remove(newarpfile);
		return(0);
	}
	if (rename(arpfile, oldarpfile) < 0) {
		lg(LOG_ERR, "rename %s -> %s: %s",
		    arpfile, oldarpfile, strerror(errno));
		(void)remove(newarpfile);
		return(0);
	}
	if (rename(newarpfile, arpfile) < 0) {
		lg(LOG_ERR, "rename %s -> %s: %s",
		    newarpfile, arpfile, strerror(errno));
		return(0);
	}
	return(1);
}

/* Initialize the databases */
int
readdata(void)
{
	FILE *f;

	if ((f = fopen(arpfile, "r")) == NULL) {
		lg(LOG_ERR, "fopen(%s): %s", arpfile, strerror(errno));
		return(0);
	}
	if (!file_loop(f, ent_add, arpfile)) {
		(void)fclose(f);
		return(0);
	}
	(void)fclose(f);

	/* It's not fatal if we can't open the ethercodes file */
	if ((f = fopen(ethercodes, "r")) != NULL) {
		(void)ec_loop(f, ec_add, ethercodes);
		(void)fclose(f);
	}

	return(1);
}

char *
savestr(const char *str)
{
	int i;
	char *cp;
	static char *strptr = NULL;
	static int strsize = 0;

	i = strlen(str) + 1;
	if (i > strsize) {
		strsize = i + 1024;
		strptr = malloc(strsize);
		if (strptr == NULL) {
			lg(LOG_ERR, "savestr(): malloc: %s", strerror(errno));
			exit(1);
		}
		memset(strptr, 0, strsize);
	}
	(void)strcpy(strptr, str);
	cp = strptr;
	strptr += i;
	strsize -= i;
	return (cp);
}
