/*
* @license Apache-2.0
*
* Copyright (c) 2021 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

// TypeScript Version: 2.0

/* tslint:disable:max-line-length */
/* tslint:disable:max-file-line-count */

import cdf = require( './../../../../../base/dists/gumbel/cdf' );
import Gumbel = require( './../../../../../base/dists/gumbel/ctor' );
import entropy = require( './../../../../../base/dists/gumbel/entropy' );
import kurtosis = require( './../../../../../base/dists/gumbel/kurtosis' );
import logcdf = require( './../../../../../base/dists/gumbel/logcdf' );
import logpdf = require( './../../../../../base/dists/gumbel/logpdf' );
import mean = require( './../../../../../base/dists/gumbel/mean' );
import median = require( './../../../../../base/dists/gumbel/median' );
import mgf = require( './../../../../../base/dists/gumbel/mgf' );
import mode = require( './../../../../../base/dists/gumbel/mode' );
import pdf = require( './../../../../../base/dists/gumbel/pdf' );
import quantile = require( './../../../../../base/dists/gumbel/quantile' );
import skewness = require( './../../../../../base/dists/gumbel/skewness' );
import stdev = require( './../../../../../base/dists/gumbel/stdev' );
import variance = require( './../../../../../base/dists/gumbel/variance' );

/**
* Interface describing the `gumbel` namespace.
*/
interface Namespace {
	/**
	* Gumbel distribution cumulative distribution function (CDF).
	*
	* @param x - input value
	* @param mu - location parameter
	* @param beta - scale parameter
	* @returns evaluated CDF
	*
	* @example
	* var y = ns.cdf( 10.0, 0.0, 3.0 );
	* // returns ~0.965
	*
	* y = ns.cdf( 0.0, 0.0, 3.0 );
	* // returns ~0.368
	*
	* var myCDF = ns.cdf.factory( 2.0, 3.0 );
	* y = myCDF( 10.0 );
	* // returns ~0.933
	*
	* y = myCDF( 2.0 );
	* // returns ~0.368
	*/
	cdf: typeof cdf;

	/**
	* Gumbel distribution.
	*/
	Gumbel: typeof Gumbel;

	/**
	* Returns the differential entropy for a Gumbel distribution with location `mu` and scale `beta`.
	*
	* ## Notes
	*
	* -   If provided `beta <= 0`, the function returns `NaN`.
	*
	* @param mu - location parameter
	* @param beta - scale parameter
	* @returns entropy
	*
	* @example
	* var y = ns.entropy( 0.0, 1.0 );
	* // returns ~1.577
	*
	* @example
	* var y = ns.entropy( 5.0, 2.0 );
	* // returns ~2.27
	*
	* @example
	* var y = ns.entropy( NaN, 1.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.entropy( 0.0, NaN );
	* // returns NaN
	*
	* @example
	* var y = ns.entropy( 0.0, 0.0 );
	* // returns NaN
	*/
	entropy: typeof entropy;

	/**
	* Returns the excess kurtosis for a Gumbel distribution with location `mu` and scale `beta`.
	*
	* ## Notes
	*
	* -   If provided `beta <= 0`, the function returns `NaN`.
	*
	* @param mu - location parameter
	* @param beta - scale parameter
	* @returns excess kurtosis
	*
	* @example
	* var y = ns.kurtosis( 0.0, 1.0 );
	* // returns 2.4
	*
	* @example
	* var y = ns.kurtosis( 5.0, 2.0 );
	* // returns 2.4
	*
	* @example
	* var y = ns.kurtosis( NaN, 1.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.kurtosis( 0.0, NaN );
	* // returns NaN
	*
	* @example
	* var y = ns.kurtosis( 0.0, 0.0 );
	* // returns NaN
	*/
	kurtosis: typeof kurtosis;

	/**
	* Gumbel distribution logarithm of cumulative distribution function (CDF).
	*
	* @param x - input value
	* @param mu - location parameter
	* @param beta - scale parameter
	* @returns evaluated logCDF
	*
	* @example
	* var y = ns.logcdf( 10.0, 0.0, 3.0 );
	* // returns ~-0.036
	*
	* y = ns.logcdf( 0.0, 0.0, 3.0 );
	* // returns ~-1
	*
	* var mylogcdf = ns.logcdf.factory( 2.0, 3.0 );
	* y = mylogcdf( 10.0 );
	* // returns ~-0.069
	*
	* y = mylogcdf( 2.0 );
	* // returns ~-1
	*/
	logcdf: typeof logcdf;

	/**
	* Gumbel distribution natural logarithm of probability density function (logPDF).
	*
	* @param x - input value
	* @param mu - location parameter
	* @param beta - scale parameter
	* @returns evaluated logPDF
	*
	* @example
	* var y = ns.logpdf( 2.0, 0.0, 1.0 );
	* // returns ~-2.137
	*
	* var mylogpdf = ns.logpdf.factory( 10.0, 2.0 );
	* y = mylogpdf( 10.0 );
	* // returns ~-1.693
	*
	* y = mylogpdf( 12.0 );
	* // returns ~-2.064
	*/
	logpdf: typeof logpdf;

	/**
	* Returns the mean for a Gumbel distribution with location `mu` and scale `beta`.
	*
	* ## Notes
	*
	* -   If provided `beta <= 0`, the function returns `NaN`.
	*
	* @param mu - location parameter
	* @param beta - scale parameter
	* @returns mean
	*
	* @example
	* var y = ns.mean( 0.0, 1.0 );
	* // returns ~0.577
	*
	* @example
	* var y = ns.mean( 5.0, 2.0 );
	* // returns ~6.154
	*
	* @example
	* var y = ns.mean( NaN, 1.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.mean( 0.0, NaN );
	* // returns NaN
	*
	* @example
	* var y = ns.mean( 0.0, 0.0 );
	* // returns NaN
	*/
	mean: typeof mean;

	/**
	* Returns the median for a Gumbel distribution with location `mu` and scale `beta`.
	*
	* ## Notes
	*
	* -   If provided `beta <= 0`, the function returns `NaN`.
	*
	* @param mu - location parameter
	* @param beta - scale parameter
	* @returns median
	*
	* @example
	* var y = ns.median( 0.0, 1.0 );
	* // returns ~0.367
	*
	* @example
	* var y = ns.median( 5.0, 2.0 );
	* // returns ~5.733
	*
	* @example
	* var y = ns.median( NaN, 1.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.median( 0.0, NaN );
	* // returns NaN
	*
	* @example
	* var y = ns.median( 0.0, 0.0 );
	* // returns NaN
	*/
	median: typeof median;

	/**
	* Gamma distribution moment-generating function (MGF).
	*
	* @param t - input value
	* @param mu - mean
	* @param beta - scale parameter
	* @returns evaluated MGF
	*
	* @example
	* var y = ns.mgf( 0.5, 0.5, 1.0 );
	* // returns ~1.414
	*
	* y = ns.mgf( 0.1, 1.0, 1.0 );
	* // returns ~1.111
	*
	* y = ns.mgf( -1.0, 4.0, 2.0 );
	* // returns ~0.198
	*
	* var mymgf = ns.mgf.factory( 3.0, 1.5 );
	*
	* y = mymgf( 1.0 );
	* // returns ~26.999
	*
	* y = mymgf( 0.5 );
	* // returns ~3.375
	*/
	mgf: typeof mgf;

	/**
	* Returns the mode for a Gumbel distribution with location `mu` and scale `beta`.
	*
	* ## Notes
	*
	* -   If provided `beta <= 0`, the function returns `NaN`.
	*
	* @param mu - location parameter
	* @param beta - scale parameter
	* @returns mode
	*
	* @example
	* var y = ns.mode( 0.0, 1.0 );
	* // returns 0.0
	*
	* @example
	* var y = ns.mode( 5.0, 2.0 );
	* // returns 5.0
	*
	* @example
	* var y = ns.mode( NaN, 1.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.mode( 0.0, NaN );
	* // returns NaN
	*
	* @example
	* var y = ns.mode( 0.0, 0.0 );
	* // returns NaN
	*/
	mode: typeof mode;

	/**
	* Gumbel distribution probability density function (PDF).
	*
	* @param x - input value
	* @param mu - location parameter
	* @param beta - scale parameter
	* @returns evaluated PDF
	*
	* @example
	* var y = ns.pdf( 2.0, 0.0, 1.0 );
	* // returns ~0.118
	*
	* var myPDF = ns.pdf.factory( 10.0, 2.0 );
	* y = myPDF( 10.0 );
	* // returns ~0.184
	*
	* y = myPDF( 12.0 );
	* // returns ~0.127
	*/
	pdf: typeof pdf;

	/**
	* Gumbel distribution quantile function.
	*
	* @param p - input value
	* @param mu - location parameter
	* @param beta - scale parameter
	* @returns evaluated quantile function
	*
	* @example
	* var y = ns.quantile( 0.8, 0.0, 1.0 );
	* // returns ~1.4999
	*
	* var myQuantile = ns.quantile.factory( 8.0, 2.0 );
	* y = myQuantile( 0.5 );
	* // returns ~8.733
	*
	* y = myQuantile( 0.7 );
	* // returns ~10.062
	*/
	quantile: typeof quantile;

	/**
	* Returns the skewness for a Gumbel distribution with location `mu` and scale `beta`.
	*
	* ## Notes
	*
	* -   If provided `beta <= 0`, the function returns `NaN`.
	*
	* @param mu - location parameter
	* @param beta - scale parameter
	* @returns skewness
	*
	* @example
	* var y = ns.skewness( 0.0, 1.0 );
	* // returns ~1.14
	*
	* @example
	* var y = ns.skewness( 5.0, 2.0 );
	* // returns ~1.14
	*
	* @example
	* var y = ns.skewness( NaN, 1.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.skewness( 0.0, NaN );
	* // returns NaN
	*
	* @example
	* var y = ns.skewness( 0.0, 0.0 );
	* // returns NaN
	*/
	skewness: typeof skewness;

	/**
	* Returns the standard deviation for a Gumbel distribution with location `mu` and scale `beta`.
	*
	* ## Notes
	*
	* -   If provided `beta <= 0`, the function returns `NaN`.
	*
	* @param mu - location parameter
	* @param beta - scale parameter
	* @returns standard deviation
	*
	* @example
	* var y = ns.stdev( 0.0, 1.0 );
	* // returns ~1.283
	*
	* @example
	* var y = ns.stdev( 5.0, 2.0 );
	* // returns ~2.565
	*
	* @example
	* var y = ns.stdev( NaN, 1.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.stdev( 0.0, NaN );
	* // returns NaN
	*
	* @example
	* var y = ns.stdev( 0.0, 0.0 );
	* // returns NaN
	*/
	stdev: typeof stdev;

	/**
	* Returns the variance for a Gumbel distribution with location `mu` and scale `beta`.
	*
	* ## Notes
	*
	* -   If provided `beta <= 0`, the function returns `NaN`.
	*
	* @param mu - location parameter
	* @param beta - scale parameter
	* @returns variance
	*
	* @example
	* var y = ns.variance( 0.0, 1.0 );
	* // returns ~1.645
	*
	* @example
	* var y = ns.variance( 5.0, 2.0 );
	* // returns ~6.58
	*
	* @example
	* var y = ns.variance( NaN, 1.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.variance( 0.0, NaN );
	* // returns NaN
	*
	* @example
	* var y = ns.variance( 0.0, 0.0 );
	* // returns NaN
	*/
	variance: typeof variance;
}

/**
* Gumbel distribution.
*/
declare var ns: Namespace;


// EXPORTS //

export = ns;
