% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/samc.R
\name{samc}
\alias{samc}
\alias{samc,SpatRaster,SpatRaster,SpatRaster,list-method}
\alias{samc,RasterLayer,RasterLayer,RasterLayer,list-method}
\alias{samc,SpatRaster,SpatRaster,missing,list-method}
\alias{samc,RasterLayer,RasterLayer,missing,list-method}
\alias{samc,matrix,matrix,matrix,list-method}
\alias{samc,matrix,matrix,missing,list-method}
\alias{samc,dgCMatrix,missing,missing,missing-method}
\alias{samc,matrix,missing,missing,missing-method}
\title{Create an samc object}
\usage{
samc(data, absorption, fidelity, model, options = NULL)

\S4method{samc}{SpatRaster,SpatRaster,SpatRaster,list}(data, absorption, fidelity, model, options = NULL)

\S4method{samc}{RasterLayer,RasterLayer,RasterLayer,list}(data, absorption, fidelity, model, options = NULL)

\S4method{samc}{SpatRaster,SpatRaster,missing,list}(data, absorption, model, options = NULL)

\S4method{samc}{RasterLayer,RasterLayer,missing,list}(data, absorption, model, options = NULL)

\S4method{samc}{matrix,matrix,matrix,list}(data, absorption, fidelity, model, options = NULL)

\S4method{samc}{matrix,matrix,missing,list}(data, absorption, model, options = NULL)

\S4method{samc}{dgCMatrix,missing,missing,missing}(data, options = NULL)

\S4method{samc}{matrix,missing,missing,missing}(data, options = NULL)
}
\arguments{
\item{data}{A \code{\link[terra]{SpatRaster-class}} or \code{\link[raster]{RasterLayer-class}} or \code{\link[base]{matrix}} or Matrix package dgCMatrix sparse matrix.}

\item{absorption}{A \code{\link[terra]{SpatRaster-class}} or \code{\link[raster]{RasterLayer-class}} or \code{\link[base]{matrix}}}

\item{fidelity}{A \code{\link[terra]{SpatRaster-class}} or \code{\link[raster]{RasterLayer-class}} or \code{\link[base]{matrix}}}

\item{model}{A list with args for constructing a transition matrix.}

\item{options}{A list of options that changes how the samc behaves computationally.}
}
\value{
A \code{\link{samc-class}} object
}
\description{
Create an samc object that contains the absorbing Markov chain data
}
\details{
This function is used to create a \code{\link{samc-class}} object. There are
multiple options for creating this object.

\strong{Option 1: Raster or Matrix Maps}

\emph{\code{samc(data = matrix, absorption = matrix, fidelity = matrix, model = list())}}

\emph{\code{samc(data = SpatRaster, absorption = SpatRaster, fidelity = SpatRaster, model = list())}}

\emph{\code{samc(data = RasterLayer, absorption = RasterLayer, fidelity = RasterLayer, model = list())}}

The \code{\link{samc-class}} object can be created from a combination of
resistance (or conductance), absorption, and fidelity data. These different landscape data
inputs must be the same type (a matrix, SpatRaster, or RasterLayer), and have identical
properties, including dimensions, location of NA cells, and CRS (if using
raster inputs).

The \code{data} and \code{absorption} inputs are always mandatory for this approach. The
\code{fidelity} input is optional. If the \code{fidelity} input is not provided, then it
is assumed that there is no site fidelity (i.e., individuals will always move
to an adjacent cell each time step).

The \code{model} parameter is a mandatory list with the following template:
\code{list(fun = `function`, dir = `numeric`, sym = `logical`)}. It is used when calculating
the values for the transition matrix. \code{fun} must be a mathematical function
taking a single vector \code{x} as input. \code{x[1]} contains the value for node \emph{i} and
\code{x[2]} contains the value for node \emph{j} from the \code{data} input to \code{samc()}.
\code{dir} determines which neighboring nodes are used andmust be either \code{4} or \code{8}.
\code{symmetric} is an optimization to reduce redundant work for when \code{fun} is communative.
It must be either \code{TRUE} or \code{FALSE}.

Special cases for \code{fun} exist to significantly speed up creation of the samc model.
They are selected using specific strings as the value for \code{fun}. Currently,
\code{"1/mean(x)"} and \code{"x[2]"} are implemented. Other special cases can be implemented
on request.

When using raster inputs, SpatRaster objects (from the terra package) are recommended
over RasterLayer objects (from the raster package). Internally, samc is using SpatRaster
objects, which means RasterLayer objects are being converted to SpatRaster objects,
which is a source of memory inefficiency.

\strong{Option 2: P Matrix}

\emph{\code{samc(data = matrix)}}

\emph{\code{samc(data = dgCMatrix)}}

The \code{data} parameter can be used alone to create a \code{\link{samc-class}} object
directly from a preconstructed P matrix. This matrix must be either a base R
matrix, or a sparse matrix (dgCMatrix format) from the Matrix package. It
must meet the following requirements:
\itemize{
  \item The number of rows must equal the number of columns (a square matrix)
  \item Total absorption must be a single column on the right-hand side of the matrix
  \item At the bottom of the matrix, there must be a row filled with 0's except
  for the last element (bottom-right of the matrix diagonal), which must be set to 1
  \item Every disconnected region of the matrix must have at least one non-zero
  absorbing value
  \item Each row must sum to 1
  \item All values must be in the range of 0-1
}

Additionally, the columns and rows of the P matrix may be named (e.g., using
dimnames(), rowname(), colnames(), etc). When specifying \code{origin} or \code{dest} inputs
to metrics, these names may be used instead of cell numbers. This has the
advantage of making the code for an analysis easier to read and interpret,
which may also help to eliminate unintentional mistakes. There are two
requirements for naming the rows/cols of a P matrix. First, since the P matrix
represents a pairwise matrix, the row and column names must be the same. Second,
there must be no duplicate names. The exception to these rules is the very last
column and the very last row of the P matrix. Since these are not part of the
pairwise transition matrix, they may have whatever names the user prefers.

\strong{Additional Options}

Additional options can be passed to \code{samc(..., options)} as a list. There
are several possible options: \code{list(method = `character`, threads = `numeric`,
override = `logical`, precision = `character`)}. The use of options varies depending
on other inputs. Some options can be changed after model creation (see samc-class
documentation). It can be omitted for default behaviors.

\code{method} controls the type of mathematical algorithm used for the model. It
can be either \code{"direct"} (the default), \code{"iter"}, or \code{"conv"}. See the
Computation Methods vignette for details.

\code{threads} is a positive integer (default \code{1}) that enables parallelization in
certain limited cases. See the Parallel Computing vignette for details.

\code{override} is a logical value (default: \code{FALSE}) that enables certain memory
intensive calculations. It only applies to the \code{"direct"} and \code{"iter"} methods.
See the samc-class reference for details.

\code{precision} controls the numerical precision of calculations. It can either be
\code{"double"} (~15 digits precision; the default) or \code{"double"} (~7 digits of
precision). Choosing the lower precision level can lead to significant speed
improvements, and reduces the memory requirements by approximately half. It
currently only applies to the \code{"conv"} method.

\strong{Additional Information}

Depending on the data used to construct the samc-class object, some metrics
may cause crashes. This is a result of the underlying P matrix having specific
properties that make some equations unsolvable. One known case is a P matrix
that represents a disconnected graph, which can lead to the \code{cond_passage()}
function crashing. In terms of raster/matrix inputs, a disconnected graph
occurs when one or more pixels/cells are unreachable from other pixels/cells
due to the presence of a full barrier made up of NA values. In a raster, these
may be obvious as islands but can be as inconspicuous as a rogue isolated
pixel. There may be other currently unknown situations that lead to unsolvable
metrics.

Future work is planned towards identifying these issues during the creation of
the samc-class object and handling them appropriately to prevent inadvertent
crashes.

\strong{Version 3 Changes}

Support for creating samc-class objects from TransitionLayer objects was removed
so that the package is not dependent on gdistance.

\strong{Version 2 Changes}

Version 1.5.0 officially removed support for the deprecated \code{resistance}, \code{tr_fun},
\code{directions}, \code{p_mat}, \code{latlon}, and \code{override} arguments. Old
code will have to be updated to the new samc() function structure in order to work.
}
\examples{
# "Load" the data. In this case we are using data built into the package.
# In practice, users will likely load raster data using the raster() function
# from the raster package.
res_data <- samc::example_split_corridor$res
abs_data <- samc::example_split_corridor$abs
init_data <- samc::example_split_corridor$init


# Make sure our data meets the basic input requirements of the package using
# the check() function.
check(res_data, abs_data)
check(res_data, init_data)

# Setup the details for a random-walk model
rw_model <- list(fun = function(x) 1/mean(x), # Function for calculating transition probabilities
                 dir = 8, # Directions of the transitions. Either 4 or 8.
                 sym = TRUE) # Is the function symmetric?


# Create a `samc-class` object with the resistance and absorption data using
# the samc() function. We use the recipricol of the arithmetic mean for
# calculating the transition matrix. Note, the input data here are matrices,
# not RasterLayers.
samc_obj <- samc(res_data, abs_data, model = rw_model)


# Convert the initial state data to probabilities
init_prob_data <- init_data / sum(init_data, na.rm = TRUE)


# Calculate short- and long-term metrics using the analytical functions
short_mort <- mortality(samc_obj, init_prob_data, time = 50)
short_dist <- distribution(samc_obj, origin = 3, time = 50)
long_disp <- dispersal(samc_obj, init_prob_data)
visit <- visitation(samc_obj, dest = 4)
surv <- survival(samc_obj)


# Use the map() function to turn vector results into RasterLayer objects.
short_mort_map <- map(samc_obj, short_mort)
short_dist_map <- map(samc_obj, short_dist)
long_disp_map <- map(samc_obj, long_disp)
visit_map <- map(samc_obj, visit)
surv_map <- map(samc_obj, surv)
}
