% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rnndescent.R
\name{graph_knn_query}
\alias{graph_knn_query}
\title{Query a search graph for nearest neighbors}
\usage{
graph_knn_query(
  query,
  reference,
  reference_graph,
  k = NULL,
  metric = "euclidean",
  init = NULL,
  epsilon = 0.1,
  max_search_fraction = 1,
  use_alt_metric = TRUE,
  n_threads = 0,
  verbose = FALSE,
  obs = "R"
)
}
\arguments{
\item{query}{Matrix of \code{n} query items, with observations in the rows and
features in the columns. Optionally, the data may be passed with the
observations in the columns, by setting \code{obs = "C"}, which should be more
efficient. The \code{reference} data must be passed in the same orientation as
\code{query}. Possible formats are \code{\link[base:data.frame]{base::data.frame()}}, \code{\link[base:matrix]{base::matrix()}}
or \code{\link[Matrix:sparseMatrix]{Matrix::sparseMatrix()}}. Sparse matrices should be in \code{dgCMatrix}
format. Dataframes will be converted to \code{numerical} matrix format
internally, so if your data columns are \code{logical} and intended to be used
with the specialized binary \code{metric}s, you should convert it to a logical
matrix first (otherwise you will get the slower dense numerical version).}

\item{reference}{Matrix of \code{m} reference items, with observations in the rows
and features in the columns. The nearest neighbors to the queries are
calculated from this data. Optionally, the data may be passed with the
observations in the columns, by setting \code{obs = "C"}, which should be more
efficient. The \code{query} data must be passed in the same format and
orientation as \code{reference}. Possible formats are \code{\link[base:data.frame]{base::data.frame()}},
\code{\link[base:matrix]{base::matrix()}} or \code{\link[Matrix:sparseMatrix]{Matrix::sparseMatrix()}}. Sparse matrices should be in
\code{dgCMatrix} format.}

\item{reference_graph}{Search graph of the \code{reference} data. A neighbor
graph, such as that output from \code{\link[=nnd_knn]{nnd_knn()}} can be used, but
preferably a suitably prepared sparse search graph should be used, such as
that output by \code{\link[=prepare_search_graph]{prepare_search_graph()}}.}

\item{k}{Number of nearest neighbors to return. Optional if \code{init} is
specified.}

\item{metric}{Type of distance calculation to use. One of:
\itemize{
\item \code{"braycurtis"}
\item \code{"canberra"}
\item \code{"chebyshev"}
\item \code{"correlation"} (1 minus the Pearson correlation)
\item \code{"cosine"}
\item \code{"dice"}
\item \code{"euclidean"}
\item \code{"hamming"}
\item \code{"hellinger"}
\item \code{"jaccard"}
\item \code{"jensenshannon"}
\item \code{"kulsinski"}
\item \code{"sqeuclidean"} (squared Euclidean)
\item \code{"manhattan"}
\item \code{"rogerstanimoto"}
\item \code{"russellrao"}
\item \code{"sokalmichener"}
\item \code{"sokalsneath"}
\item \code{"spearmanr"} (1 minus the Spearman rank correlation)
\item \code{"symmetrickl"} (symmetric Kullback-Leibler divergence)
\item \code{"tsss"} (Triangle Area Similarity-Sector Area Similarity or TS-SS
metric)
\item \code{"yule"}
}

For non-sparse data, the following variants are available with
preprocessing: this trades memory for a potential speed up during the
distance calculation. Some minor numerical differences should be expected
compared to the non-preprocessed versions:
\itemize{
\item \code{"cosine-preprocess"}: \code{cosine} with preprocessing.
\item \code{"correlation-preprocess"}: \code{correlation} with preprocessing.
}

For non-sparse binary data passed as a \code{logical} matrix, the following
metrics have specialized variants which should be substantially faster than
the non-binary variants (in other cases the logical data will be treated as
a dense numeric vector of 0s and 1s):
\itemize{
\item \code{"dice"}
\item \code{"hamming"}
\item \code{"jaccard"}
\item \code{"kulsinski"}
\item \code{"matching"}
\item \code{"rogerstanimoto"}
\item \code{"russellrao"}
\item \code{"sokalmichener"}
\item \code{"sokalsneath"}
\item \code{"yule"}
}}

\item{init}{Initial \code{query} neighbor graph to optimize. If not provided, \code{k}
random neighbors are created. If provided, the input format must be one of:
\enumerate{
\item A list containing:
\itemize{
\item \code{idx} an \code{n} by \code{k} matrix containing the nearest neighbor indices.
\item \code{dist} (optional) an \code{n} by \code{k} matrix containing the nearest neighbor
distances.
}

If \code{k} and \code{init} are specified as arguments to this function, and the
number of neighbors provided in \code{init} is not equal to \code{k} then:
\itemize{
\item if \code{k} is smaller, only the \code{k} closest values in \code{init} are retained.
\item if \code{k} is larger, then random neighbors will be chosen to fill \code{init} to
the size of \code{k}. Note that there is no checking if any of the random
neighbors are duplicates of what is already in \code{init} so effectively fewer
than \code{k} neighbors may be chosen for some observations under these
circumstances.
}

If the input distances are omitted, they will be calculated for you.
\item A random projection forest, such as that returned from \code{\link[=rpf_build]{rpf_build()}} or
\code{\link[=rpf_knn]{rpf_knn()}} with \code{ret_forest = TRUE}.
}}

\item{epsilon}{Controls trade-off between accuracy and search cost, as
described by Iwasaki and Miyazaki (2018), by specifying a distance
tolerance on whether to explore the neighbors of candidate points. The
larger the value, the more neighbors will be searched. A value of 0.1
allows query-candidate distances to be 10\% larger than the current
most-distant neighbor of the query point, 0.2 means 20\%, and so on.
Suggested values are between 0-0.5, although this value is highly dependent
on the distribution of distances in the dataset (higher dimensional data
should choose a smaller cutoff). Too large a value of \code{epsilon} will result
in the query search approaching brute force comparison. Use this parameter
in conjunction with \code{max_search_fraction} and \code{\link[=prepare_search_graph]{prepare_search_graph()}} to
prevent excessive run time. Default is 0.1. If you set \code{verbose = TRUE},
statistics of the number of distance calculations will be logged which
can help you tune \code{epsilon}.}

\item{max_search_fraction}{Maximum fraction of the reference data to search.
This is a value between 0 (search none of the reference data) and 1 (search
all of the data if necessary). This works in conjunction with \code{epsilon} and
will terminate the search early if the specified fraction of the reference
data has been searched. Default is 1.}

\item{use_alt_metric}{If \code{TRUE}, use faster metrics that maintain the
ordering of distances internally (e.g. squared Euclidean distances if using
\code{metric = "euclidean"}), then apply a correction at the end. Probably the
only reason to set this to \code{FALSE} is if you suspect that some sort of
numeric issue is occurring with your data in the alternative code path. If
a search forest is used for initialization via the \code{init} parameter, then
the metric is fetched from there and this setting is ignored.}

\item{n_threads}{Number of threads to use.}

\item{verbose}{If \code{TRUE}, log information to the console.}

\item{obs}{set to \code{"C"} to indicate that the input \code{query} and \code{reference}
orientation stores each observation as a column (the orientation must be
consistent). The default \code{"R"} means that observations are stored in each
row. Storing the data by row is usually more convenient, but internally
your data will be converted to column storage. Passing it already
column-oriented will save some memory and (a small amount of) CPU usage.}
}
\value{
the approximate nearest neighbor graph as a list containing:
\itemize{
\item \code{idx} a \code{n} by \code{k} matrix containing the nearest neighbor indices
specifying the row of the neighbor in \code{reference}.
\item \code{dist} a \code{n} by \code{k} matrix containing the nearest neighbor distances.
}
}
\description{
Run queries against a search graph, to return nearest neighbors taken from
the reference data used to build that graph.
}
\details{
A greedy beam search is used to query the graph, combining two search pruning
strategies. The first, due to Iwasaki and Miyazaki (2018), only considers
new candidates within a relative distance of the current furthest neighbor
in the query's graph. The second, due to Harwood and Drummond (2016), puts a
limit on the absolute number of distance calculations to carry out. See the
\code{epsilon} and \code{max_search_fraction} parameters respectively.
}
\examples{
# 100 reference iris items
iris_ref <- iris[iris$Species \%in\% c("setosa", "versicolor"), ]

# 50 query items
iris_query <- iris[iris$Species == "versicolor", ]

# First, find the approximate 4-nearest neighbor graph for the references:
iris_ref_graph <- nnd_knn(iris_ref, k = 4)

# For each item in iris_query find the 4 nearest neighbors in iris_ref.
# You need to pass both the reference data and the reference graph.
# If you pass a data frame, non-numeric columns are removed.
# set verbose = TRUE to get details on the progress being made
iris_query_nn <- graph_knn_query(iris_query, iris_ref, iris_ref_graph,
  k = 4, metric = "euclidean", verbose = TRUE
)

# A more complete example, converting the initial knn into a search graph
# and using a filtered random projection forest to initialize the search
# create initial knn and forest
iris_ref_graph <- nnd_knn(iris_ref, k = 4, init = "tree", ret_forest = TRUE)
# keep the best tree in the forest
forest <- rpf_filter(iris_ref_graph, n_trees = 1)
# expand the knn into a search graph
iris_ref_search_graph <- prepare_search_graph(iris_ref, iris_ref_graph)
# run the query with the improved graph and initialization
iris_query_nn <- graph_knn_query(iris_query, iris_ref, iris_ref_search_graph,
  init = forest, k = 4
)

}
\references{
Harwood, B., & Drummond, T. (2016).
Fanng: Fast approximate nearest neighbour graphs.
In \emph{Proceedings of the IEEE Conference on Computer Vision and Pattern Recognition}
(pp. 5713-5722).

Iwasaki, M., & Miyazaki, D. (2018).
Optimization of indexing based on k-nearest neighbor graph for proximity
search in high-dimensional data.
\emph{arXiv preprint arXiv:1810.07355}.
}
