% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ard_conversion.R
\name{pool_to_ard}
\alias{pool_to_ard}
\title{Convert Pool Object to ARD Format}
\usage{
pool_to_ard(pool_obj, analysis_obj = NULL, conf.level = NULL)
}
\arguments{
\item{pool_obj}{A pooled analysis object of class \code{"pool"}, typically
obtained from \code{\link[rbmi:pool]{rbmi::pool()}} after calling \code{\link[=analyse_mi_data]{analyse_mi_data()}}.}

\item{analysis_obj}{An optional analysis object (output of
\code{\link[=analyse_mi_data]{analyse_mi_data()}}), used to compute MI diagnostic statistics. When
provided and the pooling method is Rubin's rules, the ARD includes
additional stat rows for FMI, lambda, RIV, Barnard-Rubin adjusted df,
complete-data df, relative efficiency, and the number of imputations per
parameter. When \code{NULL} (the default), only the base ARD is returned.}

\item{conf.level}{Confidence level used for CI labels (e.g., \code{0.95} produces
"95\% CI Lower"). If \code{NULL} (the default), the value is taken from
\code{pool_obj$conf.level}. If that is also \code{NULL}, defaults to \code{0.95}.}
}
\value{
A data frame of class \code{"card"} (ARD format) with grouping columns
for visit (\code{group1}), parameter_type (\code{group2}), and lsm_type (\code{group3}).
Each parameter produces rows for five statistics: estimate, std.error,
conf.low, conf.high, and p.value, plus a method row. When \code{analysis_obj}
is provided and the pooling method is Rubin's rules, additional diagnostic
stat rows are included: fmi, lambda, riv, df.adjusted, df.complete, re,
and m.imputations.
}
\description{
Converts an rbmi pool object to the pharmaverse Analysis Results Dataset
(ARD) standard using the \pkg{cards} package. The ARD format is a
long-format data frame where each row represents a single statistic for a
given parameter, with grouping columns for visit, parameter type, and
least-squares-mean type.
}
\details{
The function works by:
\enumerate{
\item Tidying the pool object via \code{\link[=tidy_pool_obj]{tidy_pool_obj()}}
\item Reshaping each parameter into long-format ARD rows (one row per statistic)
\item Adding grouping columns (visit, parameter_type, lsm_type)
\item Optionally enriching with MI diagnostic statistics when \code{analysis_obj}
is provided
\item Applying \code{\link[cards:as_card]{cards::as_card()}} and \code{\link[cards:tidy_ard_order]{cards::tidy_ard_column_order()}} for
standard ARD structure
}

When \code{analysis_obj} is provided:
\itemize{
\item For Rubin's rules pooling: diagnostic statistics (FMI, lambda, RIV,
Barnard-Rubin adjusted df, relative efficiency) are computed per parameter
using the per-imputation estimates, standard errors, and degrees of freedom.
\item For non-Rubin pooling methods: an informative message is emitted and the
base ARD is returned without diagnostic rows.
}

The resulting ARD passes \code{\link[cards:check_ard_structure]{cards::check_ard_structure()}} validation and is
suitable for downstream use with \pkg{gtsummary}.
}
\examples{
\donttest{
if (requireNamespace("cards", quietly = TRUE)) {
  library(rbmi)
  data("ADMI", package = "rbmiUtils")
  ADMI$TRT <- factor(ADMI$TRT, levels = c("Placebo", "Drug A"))
  ADMI$USUBJID <- factor(ADMI$USUBJID)
  ADMI$AVISIT <- factor(ADMI$AVISIT)

  vars <- set_vars(
    subjid = "USUBJID", visit = "AVISIT", group = "TRT",
    outcome = "CHG", covariates = c("BASE", "STRATA", "REGION")
  )
  method <- method_bayes(
    n_samples = 20,
    control = control_bayes(warmup = 20, thin = 1)
  )

  ana_obj <- analyse_mi_data(ADMI, vars, method, fun = ancova)
  pool_obj <- pool(ana_obj)

  # Base ARD
  ard <- pool_to_ard(pool_obj)

  # Enriched ARD with MI diagnostics (FMI, lambda, RIV, df)
  ard_diag <- pool_to_ard(pool_obj, analysis_obj = ana_obj)
}
}

}
\seealso{
\itemize{
\item \code{\link[rbmi:pool]{rbmi::pool()}} for creating pool objects
\item \code{\link[=analyse_mi_data]{analyse_mi_data()}} for creating analysis objects
\item \code{\link[=tidy_pool_obj]{tidy_pool_obj()}} for the underlying data transformation
\item \code{\link[cards:as_card]{cards::as_card()}} and \code{\link[cards:check_ard_structure]{cards::check_ard_structure()}} for ARD validation
}
}
