## Set default choices for plot functions
## This should be user modifyable
## So, this is set within an environment (see https://r-pkgs.org/data.html#sec-data-state)
plotDescTbl_options <- new.env(parent = emptyenv())
plotDescTbl_options$plotfun_defaults <-
  tribble(~test,                                              ~plotfun,            ~args,
          ## display as jittered scatterplots
          "Kendall's rank correlation tau",                    "w.ggscatterstats", list(messages = FALSE, plot.stats = FALSE, bf.message = FALSE, point.width.jitter = 0.1, point.height.jitter = 0.1),
          "Spearman's rank correlation rho",                   "w.ggscatterstats", list(messages = FALSE, plot.stats = FALSE, bf.message = FALSE, point.width.jitter = 0.1, point.height.jitter = 0.1),
          ## display as non-jittered scatterplots
          "Pearson's product-moment correlation",              "w.ggscatterstats", list(messages = FALSE, plot.stats = FALSE, bf.message = FALSE, point.width.jitter = 0.1, point.height.jitter = 0.1),
          ## display as barplots
          "Fisher's Exact Test for Count Data",                "w.ggbarstats",     list(pairwise.display = "none", proportion.test = FALSE, bf.message = FALSE, messages = FALSE, plot.stats = FALSE, results.subtitle = FALSE),
          "Pearson's Chi-squared test",                        "w.ggbarstats",     list(pairwise.display = "none", proportion.test = FALSE, bf.message = FALSE, messages = FALSE, plot.stats = FALSE, results.subtitle = FALSE),
          "Chi-squared Test for Trend in Proportions",         "w.ggbarstats",     list(pairwise.display = "none", proportion.test = FALSE, bf.message = FALSE, messages = FALSE, plot.stats = FALSE, results.subtitle = FALSE),
          "Cochran-Armitage test for trend",                   "w.ggbarstats",     list(pairwise.display = "none", proportion.test = FALSE, bf.message = FALSE, messages = FALSE, plot.stats = FALSE, results.subtitle = FALSE),
          "Fisher's Exact Test for Count Data with simulated p-value", "w.ggbarstats", list(pairwise.display = "none", proportion.test = FALSE, bf.message = FALSE, messages = FALSE, plot.stats = FALSE, results.subtitle = FALSE),
          "Fisher's Exact Test for Count Data with simulated p-value\n\t (based on 1e+05 replicates)", "w.ggbarstats", list(pairwise.display = "none", proportion.test = FALSE, bf.message = FALSE, messages = FALSE, plot.stats = FALSE, results.subtitle = FALSE),
          ## display as boxplots with mean annotated
          "Analysis of Variance",                              "w.ggbetweenstats", list(pairwise.display = "none", bf.message = FALSE, messages = FALSE, plot.stats = FALSE, type = "parametric", var.equal = TRUE, results.subtitle = FALSE),
          "Welch Two Sample t-test",                           "w.ggbetweenstats", list(pairwise.display = "none", bf.message = FALSE, messages = FALSE, plot.stats = FALSE, type = "parametric", var.equal = TRUE, results.subtitle = FALSE),
          "Kruskal-Wallis rank sum test",                      "w.ggbetweenstats", list(pairwise.display = "none", bf.message = FALSE, messages = FALSE, plot.stats = FALSE, type = "nonparametric", results.subtitle = FALSE),
          "Wilcoxon rank sum test with continuity correction", "w.ggbetweenstats", list(pairwise.display = "none", bf.message = FALSE, messages = FALSE, plot.stats = FALSE, type = "nonparametric", results.subtitle = FALSE),
          "Rank based test for indpendent groups",             "w.ggbetweenstats", list(pairwise.display = "none", bf.message = FALSE, messages = FALSE, plot.stats = FALSE, type = "nonparametric", results.subtitle = FALSE),
          "Rank based test for independent groups",            "w.ggbetweenstats", list(pairwise.display = "none", bf.message = FALSE, messages = FALSE, plot.stats = FALSE, type = "nonparametric", results.subtitle = FALSE),
          "Pseudo-rank based test for indpendent groups",      "w.ggbetweenstats", list(pairwise.display = "none", bf.message = FALSE, messages = FALSE, plot.stats = FALSE, type = "nonparametric", results.subtitle = FALSE),
          "Pseudo-rank based test for independent groups",     "w.ggbetweenstats", list(pairwise.display = "none", bf.message = FALSE, messages = FALSE, plot.stats = FALSE, type = "nonparametric", results.subtitle = FALSE),
          "Brunner Munzel test",                               "w.ggbetweenstats", list(pairwise.display = "none", bf.message = FALSE, messages = FALSE, plot.stats = FALSE, type = "nonparametric", results.subtitle = FALSE),
          "Studentized permutation test",                      "w.ggbetweenstats", list(pairwise.display = "none", bf.message = FALSE, messages = FALSE, plot.stats = FALSE, type = "nonparametric", results.subtitle = FALSE),
          "Jonckheere-Terpstra test",                          "w.ggbetweenstats", list(pairwise.display = "none", bf.message = FALSE, messages = FALSE, plot.stats = FALSE, type = "nonparametric", results.subtitle = FALSE),
          ## do not display
          "Watson-Williams test for homogeneity of means",     "w.emptyplot",      list(NULL),
          "<untested>",                                        "w.emptyplot",      list(NULL))

##' Report default plot functions
##' @return \code{tibble} with the currently set default suggestions for plotting function
##' @author Dr. Andreas Leha
##' @export
##' @examples
##' get_plotfun_defaults()
get_plotfun_defaults <- function() {
  plotDescTbl_options$plotfun_defaults
}

##' Change default plot functions
##' @param df \code{data.frame} with default choices for plotting functions.  See \code{\link{get_plotfun_defaults}} for the structure.
##' @return invisible. \code{tibble} with the old (prior to the change) set default suggestions for plotting function
##' @author Dr. Andreas Leha
##' @export
##' @examples
##' current <- get_plotfun_defaults()
##' current$plotfun[current$test == "Fisher's Exact Test for Count Data"] <- "emptyplot"
##' current$args[current$test == "Fisher's Exact Test for Count Data"] <- list(NULL)
##' set_plotfun_defaults(current)
set_plotfun_defaults <- function(df) {
  old <- plotDescTbl_options$plotfun_defaults
  plotDescTbl_options$plotfun_defaults <- df
  invisible(old)
}


##' Function suggesting a default plot for a given parameter in a descriptive table
##'
##' The plot suggestion is based on the test that was used for that variable.
##'
##' The default choice is taken from \code{plotDescTbl_options$plotfun_defaults} with the current associations.
##'
##' If no test was performed than numeric variables are treated as if
##' tested via t-tests and non-numeric variables as if tested via
##' Fisher test.
##' @param ltest character vector of tests. This normally filled by \code{\link[descsuppR]{buildDescrTbl}} which in turn usually uses the \code{method} element of the test result.
##' @return tibble with columns \code{plotfun} an \code{args} where
##' \code{plotfun} is a string with the name of the function to be called to generate the plot and
##' \code{args} is a list of additional (besides the data) arguments
##' @export
##' @examples
##' suggestPlot("Fisher's Exact Test for Count Data")
##'
##' suggestPlot("foobar")
##' @author Dr. Andreas Leha
suggestPlot <- function(ltest) {
  res <-
    if (ltest %in% plotDescTbl_options$plotfun_defaults$test) {
      plotDescTbl_options$plotfun_defaults %>% dplyr::filter(.data$test == ltest)
    } else {
      warning("no default plotting function for ", ltest, ". Will treat as if untested and generate an empty plot")
      plotDescTbl_options$plotfun_defaults %>% dplyr::filter(.data$test == "<untested>")
    }
  return(res)
}
