% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bqr.svy.R
\name{bqr.svy}
\alias{bqr.svy}
\title{Bayesian quantile regression for complex survey data}
\usage{
bqr.svy(
  formula,
  weights = NULL,
  data = NULL,
  quantile = 0.5,
  method = c("ald", "score", "approximate"),
  prior = NULL,
  niter = 50000,
  burnin = 0,
  thin = 1,
  verbose = TRUE,
  estimate_sigma = FALSE
)
}
\arguments{
\item{formula}{a symbolic description of the model to be fit.}

\item{weights}{an optional numerical vector containing the survey weights. If \code{NULL}, equal weights are used.}

\item{data}{an optional data frame containing the variables in the model.}

\item{quantile}{numerical scalar or vector containing quantile(s) of interest (default=0.5).}

\item{method}{one of \code{"ald"}, \code{"score"} and \code{"approximate"} (default=\code{"ald"}).}

\item{prior}{a \code{bqr_prior} object of class "prior". If omitted, a vague prior is assumed (see \code{\link{prior}}).}

\item{niter}{number of MCMC draws.}

\item{burnin}{number of initial MCMC draws to be discarded.(default = 0)}

\item{thin}{thinning parameter, i.e., keep every keepth draw (default=1).}

\item{verbose}{logical flag indicating whether to print progress messages (default=TRUE).}

\item{estimate_sigma}{logical flag indicating whether to estimate the scale parameter
when method = "ald" (default=FALSE and \eqn{\sigma^2} is set to 1)}
}
\value{
An object of class \code{"bqr.svy"}, containing:
\item{beta}{Posterior mean estimates of regression coefficients.}
\item{draws}{Posterior draws from the MCMC sampler.}
\item{accept_rate}{Average acceptance rate (if available).}
\item{warmup, thin}{MCMC control parameters used during sampling.}
\item{quantile}{The quantile(s) fitted.}
\item{prior}{Prior specification used.}
\item{formula, terms, model}{Model specification details.}
\item{runtime}{Elapsed runtime in seconds.}
\item{method}{Estimation method}
\item{estimate_sigma}{Logical flag indicating whether the scale parameter
\eqn{\sigma^2} was estimated (\code{TRUE}) or fixed at 1 (\code{FALSE}).}
}
\description{
\code{bqr.svy} implements Bayesian methods for estimating quantile regression models
for complex survey data analysis regarding single (univariate) outputs. To
improve computational efficiency, the Markov Chain Monte Carlo (MCMC) algorithms
are implemented in 'C++'.
}
\details{
The bqr.svy function can estimate three types of models, where the quantile regression
coefficients are defined at the super-population level, and their estimators are built upon
the survey weights.
\itemize{
\item \code{"ald"} – The asymmetric Laplace distribution as working likelihood.
\item \code{"score"} – A score based likelihood function.
\item \code{"approximate"} – A pseudolikelihood function based on a Gaussian approximation.
}
}
\examples{
\donttest{
# Generate population data
set.seed(123)
N    <- 10000
x1_p <- runif(N, -1, 1)
x2_p <- runif(N, -1, 1)
y_p  <- 2 + 1.5 * x1_p - 0.8 * x2_p + rnorm(N)

# Generate sample data
n <- 500
z_aux <- rnorm(N, mean = 1 + y_p, sd = .5)
p_aux <- 1 / (1 + exp(2.5 - 0.5 * z_aux))
s_ind <- sample(1:N, n, replace = FALSE, prob = p_aux)
y_s   <- y_p[s_ind]
x1_s  <- x1_p[s_ind]
x2_s  <- x2_p[s_ind]
w     <- 1 / p_aux[s_ind]
data  <- data.frame(y = y_s, x1 = x1_s, x2 = x2_s, w = w)

# Basic usage with default method ('ald') and priors (vague)
fit1 <- bqr.svy(y ~ x1 + x2, weights = w, data = data)

# Specify informative priors
prior <- prior(
  beta_x_mean = c(2, 1.5, -0.8),
  beta_x_cov  = diag(c(0.25, 0.25, 0.25)),
  sigma_shape = 1,
  sigma_rate  = 1
)
fit2 <- bqr.svy(y ~ x1 + x2, weights = w, data = data, prior = prior)

# Specify different methods
fit_score  <- bqr.svy(y ~ x1 + x2, weights = w, data = data, method = "score")
fit_approx <- bqr.svy(y ~ x1 + x2, weights = w, data = data, method = "approximate")
}

}
\references{
Nascimento, M. L. & \enc{Gonçalves}{Goncalves}, K. C. M. (2024).
Bayesian Quantile Regression Models for Complex Survey Data Under Informative Sampling.
\emph{Journal of Survey Statistics and Methodology}, 12(4), 1105–1130.
\url{doi:10.1093/jssam/smae015}
}
