% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/adrf.R
\name{adrf}
\alias{adrf}
\alias{adrf.default}
\title{Estimate an average dose-response function (ADRF)}
\usage{
adrf(x, ...)

\method{adrf}{default}(
  x,
  treat,
  vcov = "unconditional",
  cluster = NULL,
  type = "response",
  data = NULL,
  subset = NULL,
  by = NULL,
  wts = NULL,
  range = 0.95,
  n = 51,
  fwb.args = list(),
  ...
)
}
\arguments{
\item{x}{a fitted model object (e.g., from \code{\link[=lm]{lm()}} or \code{\link[=glm]{glm()}}).}

\item{...}{further arguments passed to \code{\link[marginaleffects:get_predict]{marginaleffects::get_predict()}}.}

\item{treat}{a string specifying the name of the treatment variable.}

\item{vcov}{how the covariance matrix of the estimates should be computed. If \code{"unconditional"} (the default for frequentist models), use the sandwich estimator including sampling uncertainty. If \code{"boot"} or \code{"fwb"}, use the traditional or fractional weighted bootstrap, respectively (both of which require the \CRANpkg{fwb} package to be installed). Otherwise, may be a covariance matrix or other allowed input to the \code{vcov} argument of \code{\link[marginaleffects:get_vcov]{marginaleffects::get_vcov()}}. Can also be \code{"none"} to avoid computing the uncertainty. For Bayesian models, only \code{"posterior"}, which uses the posterior of the estimates, and \code{"none"} are allowed. For models fit to multiply imputed data, \code{"boot"} and \code{"fwb"} are not allowed.}

\item{cluster}{an optional data frame or one-sided formula with the clustering terms for cluster-robust inference.}

\item{type}{character string indicating the type of prediction. Passed to \code{\link[marginaleffects:get_predict]{marginaleffects::get_predict()}}. Default is \code{"response"} for predictions on the scale of the outcome variable. Other options might include \code{"link"} for the linear predictor. This argument is ignored for \code{lm} objects.}

\item{data}{an optional data frame containing the observations originally used to fit the outcome model supplied to \code{x}. This should only be used if the supplied model is not supported by \pkg{insight}. In most cases, this should not need to be supplied.}

\item{subset}{an optional logical expression indicating the subset of data to use for estimation. Will be evaluated in the environment of the original dataset supplied to the model fitting function.}

\item{by}{optional variable(s) over which to group the estimation. Can be a character vector or one-sided formula.}

\item{wts}{optional numeric vector of weights to generalize the effect curve to a weighted population.}

\item{range}{numeric; a numeric vector corresponding either to the lower and upper bounds of the treatment values for which to compute the affect curve or a single number corresponding to the middle quantile of the treatment. Default is .95 to use the .025 and .975 quantiles of the treatment. See Details.}

\item{n}{integer specifying the number of equally spaced grid points on which to compute the effect curve anchor points. Default is 51; higher numbers increase computation time and size of the resulting object but improve accuracy.}

\item{fwb.args}{an optional list of arguments to be passed to \pkgfun{fwb}{fwb} when \code{vcov} is \code{"boot"} or \code{"fwb"}.}
}
\value{
An object of class \code{effect_curve}. This object is a function with attributes. See \code{\linkS4class{effect_curve}} for details on this function and its outputs.
}
\description{
Estimates the average dose-response function (ADRF) for a fitted model object.
}
\details{
\code{adrf()} estimates the ADRF by computing average predicted outcomes in the sample for counterfactual treatment values, optionally stratified by grouping variables and accounting for estimation uncertainty via unconditional or conditional variance estimation or bootstrapping. Unconditional variance estimation and bootstrapping treat the sample as random. When \code{vcov = "unconditional"}, the variance is computed using the formulas in Hansen et al. (2024), which involves augmenting the influence function with a term to account for sampling from the superpopulation. Unconditional variance estimation requires \code{\link[sandwich:estfun]{sandwich::estfun()}} and \code{\link[sandwich:bread]{sandwich::bread()}} methods for the supplied object to be available.

When a \code{mira} object from \pkg{mice} or a \code{mimira} object from \pkg{MatchThem} is supplied, analyses are applied to each imputed dataset and pooled using Rubin's rules. Bootstrapping is not allowed with such objects.

When a \code{svyglm} object from \pkg{survey} is supplied, \code{adrf()} automatically incorporates the survey weights extracted from the object. The same is true for \code{glm_weightit} objects, etc., from \pkg{WeightIt} when \code{s.weights} are supplied in the original call to \code{weightit()}. See \code{vignette("adrftools")} for more details on using the \code{wts} argument.
\subsection{\code{range}}{

The \code{range} argument controls for which range of the treatment the effect curve is to be evaluated. It can be supplied either as two numbers corresponding to the lower and upper bounds for the treatment (e.g., \code{range = c(0, 10)}) or as a single number corresponding to the middle quantile of the treatment (e.g., \code{range = .9}, which uses the .05 and .95 quantiles of the treatment as the bounds). The default is .95 to use the .025 and .975 quantiles of the treatment. When supplied as a quantile, the quantiles are evaluated incorporating the weights supplied to \code{wts}.

A reason not to use the full treatment range (e.g., by setting \code{range = 1}) is that there is likely very little certainty about the effect curve at the treatment extremes. This uncertainty can muddy tests of the effect curve. However, limiting the treatment range means inferences about the effect curve are less generalizable to more extreme values of the treatment. Note that this does not change the data used to fit the effect curve, just the points along the effect curve for which inference and estimation are to take place.
}
}
\examples{
data("nhanes3lead")

fit <- lm(Math ~ poly(logBLL, 5) *
            Male * (Age + Race + PIR +
                      Enough_Food),
          data = nhanes3lead)

# ADRF of logBLL on Math, unconditional
# inference
adrf1 <- adrf(fit, treat = "logBLL")

adrf1

## Plot the ADRF
plot(adrf1)

## ADRF estimates at given points
adrf1(logBLL = c(0, 1, 2)) |>
  summary()

# ADRF of logBLL on Math, unconditional
# inference; manual range
adrf2 <- adrf(fit, treat = "logBLL",
              range = c(0, 2))

adrf2

plot(adrf2)
\dontshow{if (rlang::is_installed("fwb")) withAutoprint(\{ # examplesIf}
# ADRF of logBLL on Math, bootstrap
# inference
\donttest{adrf_b <- adrf(fit, treat = "logBLL",
               vcov = "fwb")

adrf_b

plot(adrf_b)}
\dontshow{\}) # examplesIf}
# ADRF in subset
adrf_m <- adrf(fit, treat = "logBLL",
               subset = Male == 1)

adrf_m

# ADRFs in subgroups
adrf_by <- adrf(fit, treat = "logBLL",
                by = ~Male)

adrf_by
}
\seealso{
\itemize{
\item \code{\link[=plot.effect_curve]{plot.effect_curve()}} for plotting the ADRF
\item \code{\link[=summary.effect_curve]{summary.effect_curve()}} for testing hypotheses about the ADRF
\item \code{\link{effect_curve}} for computing point estimates along the ADRF
\item \code{\link[=curve_projection]{curve_projection()}} for projecting a simpler model onto the ADRF
\item \code{\link[=reference_curve]{reference_curve()}} for computing the difference between each point on the ADRF and a specific reference point
\item \code{\link[=curve_contrast]{curve_contrast()}} for contrasting ADRFs computed within subgroups
\item \code{\link[=amef]{amef()}} for computing the average marginal effect function (AMEF), the derivative of the ADRF
\item \code{\link[marginaleffects:predictions]{marginaleffects::avg_predictions()}} for computing average adjusted predictions for fitted models (similar to the ADRF)
}
}
