#' Non Euclidean Algorithm to Cluster
#'
#'
#' We give initial centers, calculate the distance between each point and each center and assign each point to the center with minimum distance. Calculate the center of the group and repeat the process. The process is stopped when the distance between a center and the previous one is small than COTA or the maximum number of iterations is reached.
#'
#'
#' @param data Matrix with \code{dim(data)[1]} points of \code{dim(data)[2]} dimensions.
#' @param distance Function. This function designs how the distance is going to be calculated. It must have as input two vectors and as output the distance of these vectors.
#' @param k Number. Number of groups into which we are going to group the different points.
#' @param centers_function Function. This function designs how the centers of the groups will be calculated. It must have as input \code{data} and \code{grouping} and as output a matrix that has the centers. This matrix will have as many rows as centers. With \code{grouping} we mean a list. The list component i has a vector with the numbers of the row of the matrix \code{data} where the points belonging to group i are.
#' @param init_centers Function. This function designs how we are going to calculate the initial centers. The input must be the \code{data}, \code{distance} and \code{k} and the output must be a matrix where each row has the center of one group.
#' @param seed Number. Number to fix a seed and be able to reproduce your results.
#' @param ITER Number. Maximum number of iterations.
#' @param COTA Number. The process is stopped when the distance between a center and the previous one is smaller than COTA.
#'
#' @return Returns a list with:
#'
#' \itemize{
#' \item FHW_output; is a list with
#'      \itemize{
#'   \item centers: the information of the centers updated. Matrix with \code{dim(centers)[1]} centers of \code{dim(centers)[2]} dimensions.
#'   \item grouping: the information of the groups updated. List. Each component of the list contains a vector with the points that belong to that group. More specifically, the list component i has a vector with the numbers of the row of the matrix \code{data} where the points belonging to group i are.
#'    }
#'  \item Stop_Criteria: returns the distance between one center and the previous one for all the iterations
#'  \item Chanche_yes_no: matrix, in the position \code{[i,j]} returns "yes" if the point i have changed its group in the iteration j and return "no" if the point have not changed.
#'  \item all_output: is a list with the information of the center and the groups of each iteration of the process
#'
#'    }
#'
#'
#'
#' @examples
#'set.seed(451)
#'data=rbind(matrix(runif(20,1,5), nrow = 2, ncol = 10),
#'           matrix(runif(20,20,30), nrow = 2, ncol = 10),
#'           matrix(runif(20,50,70), nrow = 2, ncol = 10))
#'k=3
#'seed=5
#'
#'
#'o2=NEC(data,
#'       RelativeDistance,
#'       k,
#'       centers_function_RelativeDistance,
#'       init_centers_random,
#'       seed=seed,
#'       10,
#'       0.01)
#'
#'
#' @export




NEC<-function(data, distance, k, centers_function, init_centers, seed=NULL, ITER, COTA){

  data=as.matrix(data)
  colnames(data)<-NULL
  rownames(data)<-NULL

  Chanche_yes_no=matrix("No", dim(data)[1], ITER)
  #Initilisng elements that we will use later
  all_output=list()

  d_centers=matrix(0,ITER-1,k)
  colnames(d_centers)<-paste0("Group_", c(1:k))
  names_col=rep(0,ITER-1)


  SumDistEnIter=rep(0, ITER)

  SumDistEntrIter=rep(0, ITER-1)

  #Just in case it is necessary
  if(!is.null(seed)){
    set.seed(seed)
  }

  #Initialize centers
  centers=init_centers(data,distance,k,centers_function)


  #Distance between each point and all the centers
  Dist_e_cent=matrix(0,dim(data)[1],dim(centers)[1])
  for (i in 1:(dim(data)[1])){
    for (j in 1:(dim(centers)[1])){
      Dist_e_cent[i,j]=distance(data[i,],centers[j,])
    }
  }

  #We obtain the most near and second most near center  for each point
  Ic12=Dist_IC1_IC2(Dist_e_cent)
  Ic12_change=Ic12
  #We assign each point to the most near center
  Group=Ic12[,1]
  grouping<-list()
  for(i in 1:(max(Group))){
    grouping[[i]]=which(Group==i)#list with the number of the rows of the data matrix that are in the group [[i]]
  }

  #If some group has no points
  if(any(sapply(grouping, function(x) length(x) == 0))==TRUE){

    EmptyGroups=which(sapply(grouping, function(x) length(x) == 0))

    np=rep(0,length(EmptyGroups) )
    old_group=rep(0,length(EmptyGroups) )
    tamOld=rep(0,length(EmptyGroups) )
    for (i in 1:length(EmptyGroups)){
      np[i]=which.min(Dist_e_cent[,EmptyGroups[i]])
      old_group[i]=encontrar_componente(grouping, np[i])
      tamOld[i]=length(grouping[[old_group[i]]])

    }

    if(all(!(old_group %in% EmptyGroups))==TRUE & any(tamOld==1)==FALSE){

      for (i in 1:length(EmptyGroups)){
        grouping[[old_group[i]]]=grouping[[old_group[i]]][grouping[[old_group[i]]]!=np[i]]
        grouping[[EmptyGroups[i]]]=np[i]
      }
    }else{
      leng_groups=sapply(grouping, length)
      PossibleGropus=which(leng_groups>length( EmptyGroups)+1)

      m=round(dim(data)[1]/k, 0)-1

      grouping[[1]]=c(1:m)

      for(i in 2:(k-1)){
        grouping[[i]]=c((m*(i-1)+1):(m*(i-1)+1+m-1))
      }

      grouping[[k]]=c((m*(k-1-1)+1+m-1+1):dim(data)[1])
    }


  }


  FHW_output=list(centers=centers, grouping=grouping)
  all_output[[1]]=FHW_output
  Chanche_yes_no[,1]=rep("Yes", dim(data)[1])

  #SumDistEnIter[1]=sum(ECDentroCluster3(data, FHW_output, distance))


  for(iter in 2:ITER){

    #Update the clusters centers.
    centers=centers_function(data, grouping)


    Dist_e_cent=matrix(0,dim(data)[1],dim(centers)[1])
    for (ii in 1:dim(data)[1]){

            for (j in 1:(dim(centers)[1])){
              Dist_e_cent[ii,j]=distance(data[ii,],centers[j,])
            }

            new_group=which.min(Dist_e_cent[ii,])
            old_group=encontrar_componente(grouping, ii)

            if(new_group!=old_group &&  length(grouping[[old_group]])!=1){
              grouping[[old_group]]=grouping[[old_group]][grouping[[old_group]]!=ii]
              grouping[[new_group]]=add_unique_numbers2(grouping[[new_group]], ii)
              Chanche_yes_no[ii, iter]="Yes"
            }
    }


    FHW_output=list(centers=centers, grouping=grouping)
    all_output[[iter]]=FHW_output



    #STOP CRITERA, de distance between centers is small than COTA
    for (j in 1:k){
      d_centers[iter-1,j]=distance(  all_output[[iter]]$centers[j,] ,    all_output[[iter-1]]$centers[j,]   )
      names_col[iter-1]=paste0("Distance between ", iter-1, " and ", iter)
    }

    if(all(d_centers[iter-1,] < COTA) ){
          rownames(d_centers)<-names_col
      return(list(FHW_output=FHW_output,Stop_Criteria=d_centers,Chanche_yes_no=Chanche_yes_no,SumDistEntrIter=SumDistEntrIter,SumDistEnIter=SumDistEnIter,all_output=all_output))
    }


  }

 warning("The algorithm is stopped because the maximum number of iterations has been reached.")
  rownames(d_centers)<-names_col
  return(list(FHW_output=FHW_output,Stop_Criteria=d_centers, Chanche_yes_no=Chanche_yes_no,all_output=all_output))


}
