% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ot.R
\name{ot_cost_sgn}
\alias{ot_cost_sgn}
\title{Compute optimal transport costs for signed measures}
\usage{
ot_cost_sgn(mu, nu, costm, mode = c("all", "diag"))
}
\arguments{
\item{mu}{matrix (row-wise) or list containing \eqn{K_1} vectors of length \eqn{N}.}

\item{nu}{matrix (row-wise) or list containing \eqn{K_2} vectors of length \eqn{N} or \code{NULL}.}

\item{costm}{cost matrix \eqn{c \in \mathbb{R}^{N \times N}}.}

\item{mode}{controls which of the pairwise OT costs are computed.}
}
\value{
The OT cost between the vectors in \code{mu} and \code{nu}.

For \code{mode = "all"} the whole matrix of size \eqn{K_1 \times K_2} is returned. If \code{mu} or \code{nu} is a vector, then this matrix is also returned as a vector.
\code{nu = NULL} means that \code{nu = mu} and only the lower triangular part is actually computed and then reflected.

If \code{mode = "diag"}, then only the diagonal is returned (requiring \eqn{K_1 = K_2}).
}
\description{
Compute the optimal transport (OT) cost between signed measures that have the same total mass.
}
\details{
The extended OT functional for vectors \eqn{\mu,\,\nu \in \mathbb{R}^N} with \eqn{\sum_{i=1}^N \mu_i = \sum_{i=1}^N \nu_i} is defined as
\deqn{
 \mathrm{OT}^{\pm}_c(\mu, \nu) := \mathrm{OT}_c(\mu^+ + \nu^-, \, \nu^+ + \mu^-)\,,
}
where \eqn{\mu^+ = \max(0, \mu)} and \eqn{\mu^- = -\min(0, \mu)} denote the positive and negative part of \eqn{\mu}, and \eqn{\mathrm{OT}_c}
is the standard OT functional. To compute the standard OT, the function \code{\link[transport:transport]{transport::transport}} is used.
The values may be computed in parallel via \code{\link[future:plan]{future::plan}}.
}
\examples{

# enable parallel computation
if (requireNamespace("future")) {
    future::plan(future::multisession)
}

# generate random signed measures with total mass 0 (row-wise)
rsum0 <- \(K, N) {
    x <- runif(K * N) |> matrix(K, N)
    x <- sweep(x, 1, rowSums(x) / N, "-")
    x[, 1] <- x[, 1] - rowSums(x)
    x
}

K1 <- 3
K2 <- 2
N <- 4
costm <- cost_matrix_lp(1:N)

set.seed(123)
mu <- rsum0(K1, N)
nu <- rsum0(K2, N)

print(ot_cost_sgn(mu[2, ], nu[2, ], costm))

# mode = "diag" requires K1 = K2
print(ot_cost_sgn(mu[1:2, ], nu, costm, mode = "diag"))

print(ot_cost_sgn(mu, nu, costm))

# only works properly if costm is semi-metric
print(ot_cost_sgn(mu, NULL, costm))
# but it requires less computations than
print(ot_cost_sgn(mu, mu, costm))
\dontshow{
## R CMD check: make sure any open connections are closed afterward
if (requireNamespace("future") && !inherits(future::plan(), "sequential")) future::plan(future::sequential)
}
}
\seealso{
\code{\link[transport:transport]{transport::transport}}
}
