
#' @title Test equality of probability vectors
#' @description Perform optimal transport (OT) barycenter based tests for equality of probability vectors in a one-way layout.
#' @param samples matrix (row-wise) or nested list containing \eqn{K} count vectors. A count vector is a vector of length \eqn{N}
#' that contains the number of times a sample was observed at the respective points.
#' @param costm semi-metric cost matrix \eqn{c \in \mathbb{R}^{N \times N}}.
#' @param null.mu probability measures \eqn{\mu} underlying the null distribution. Must be of the same structure as `samples`.
#' @param w weight vector \eqn{w \in \mathbb{R}^K_+}.
#' @param num.sim number of samples to draw from the limiting null distribution.
#' @param solver the LP solver to use, see [`ot_test_lp_solver`].
#' @param is.metric value indicating whether \eqn{c} is a metric cost matrix, see [`is_metric_cost_mat`].
#' @param verbose logical value indicating whether additional information should be printed.
#' @returns An object of class `"ot_barycenter_test"` containing:
#' \tabular{ll}{
#'  `mu`           \tab empirical version of \eqn{\mu} that is based on `samples` \cr
#'  `n`            \tab the sample sizes \cr
#'  `p.value`      \tab the \eqn{p}-value \cr
#'  `statistic`    \tab the value of the test statistic \cr
#'  `null.samples` \tab samples drawn from the null distribution \cr
#' }
#' @details Denote with \eqn{\mu^1, \ldots, \mu^K} the probability measures that underlie the samples contained in `samples`. To test for
#' the one-way null hypothesis \eqn{H_0 : \mu^1 = \ldots = \mu^K}, this test employs the OT barycenter statistic
#' which is defined as
#' \eqn{
#'  T^B(\mu) := \sqrt{\rho_n} B_c^w(\mu^1, \ldots, \mu^K)\,,
#' }
#' where \eqn{\rho_n} is a scaling factor and \eqn{B_c^w} is the OT barycenter functional, see [`ot_barycenter`].
#'
#' The test is based on the asymptotic distribution of \eqn{T^B} under under the null, for more details see the reference.
#'
#' These simulations can be done in parallel via [`future::plan`] and the progress can be shown with [`progressr::with_progress`].
#'
#' Especially for large \eqn{N} and \eqn{K}, simulating a sufficient number of samples from the limiting null distribution might take a while.
#' Consider using [`FDOTT`] instead.
#' @references TODO
#' @example examples/bary.R
#' @seealso [`FDOTT`] with `H0 = "="`.
#' @export
ot_barycenter_test <- \(samples, costm, null.mu = NULL, w = NULL,
                        num.sim = 1000, solver = ot_test_lp_solver(),
                        is.metric = is_metric_cost_mat(costm, tol.ti = Inf), verbose = FALSE) {

    print_info <- print_info_base("ot_barycenter_test", verbose)

    print_info("Checking input")
    samplemat <- get_fac_mat(samples, TRUE)
    check_samplemat(samplemat)

    K <- nrow(samplemat)
    N <- ncol(samplemat)

    check_cost_mat(costm, N, is.metric)

    n <- get_n(samplemat)
    mu <- get_mu(samplemat, n)

    print_info("Identified N = %d and K = %d", N, K)

    if (is.null(w)) {
        print_info("Generating default weights")
        w <- rep(1 / K, K)
    }

    if (is.null(null.mu)) {
        null.mu <- mu
    }
    if (is.null(dim(null.mu))) {
        null.mu <- matrix(null.mu, K, N, byrow = TRUE)
    }

    print_info("Calculating limit coefficents")
    tmp <- limit_coeffs(n)
    rho <- tmp$rho
    delta <- tmp$delta

    print_info("Calculating test statistic")
    tstat <- sqrt(rho) * ot_barycenter(mu, costm, w, solver = solver)$cost

    print_info("Sampling from the limiting distribution for %d times", num.sim)
    ls <- simulate_limit_ot_barycenter_test_null(null.mu, costm, delta = delta, w = w, num.sim = num.sim, solver = solver)

    print_info("Calculating p-value")
    p <- sum(ls >= tstat)  / num.sim

    print_info("Returning results")
    list(
        mu           = mu,
        n            = n,
        null.mu      = null.mu,
        costm        = costm,
        p.value      = p,
        statistic    = tstat,
        null.samples = ls
    ) |> structure(class = "ot_barycenter_test")
}

#' @export
print.ot_barycenter_test <- \(x, ...) {
    K <- nrow(x$mu)
    N <- ncol(x$mu)
    cat("\n")
    cat("Optimal Transport Barycenter Based Test in One-Way Layout\n")
    cat("\n")
    catf("data: mu containing K = %d probability vectors of size N = %d\n", K, N)
    cat("null hypothesis: mu^1 = ... = mu^K\n")
    catf("p-value: %f\n", x$p.value)
    cat("\n")
    invisible(x)
}
