% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ModelDiagnostics.R
\name{model_diagnostics}
\alias{model_diagnostics}
\title{DI specific wrapper of diagnostics plots for regression models with compositional predictors}
\usage{
model_diagnostics(
  model,
  which = c(1, 2, 3, 5),
  prop = NULL,
  FG = NULL,
  npoints = 3,
  cook_levels = c(0.5, 1),
  pie_radius = 0.2,
  pie_colours = NULL,
  only_extremes = FALSE,
  label_size = 4,
  points_size = 3,
  plot = TRUE,
  nrow = 0,
  ncol = 0
)
}
\arguments{
\item{model}{A statistical regression model object fit using \code{lm},
\code{glm}, \code{nlme} functions, etc.}

\item{which}{A subset of the numbers 1 to 6, by default 1, 2, 3, and 5,
referring to \cr
1 - "Residuals vs Fitted", aka "Tukey-Anscombe" plot \cr
2 - "Normal Q-Q" plot, an enhanced qqnorm(resid(.)) \cr
3 - "Scale-Location" \cr
4 - "Cook's distance" \cr
5 - "Residuals vs Leverage" \cr
6 - "Cook's dist vs Lev./(1-Lev.)" \cr
\emph{Note: If the specified model object does not inherit the \code{lm}
class, it might not be possible to create all diagnostics plots.
In these cases, the user will be notified about any plots which
can't be created.}}

\item{prop}{A character vector giving names of columns containing
proportions to show in the pie-glyphs. If not specified,
black points (geom_point) will be shown for each observation in
the model. Note: \verb{\code{prop}} can be left blank and will be
interpreted if model is a \code{Diversity-Interactions (DI)}
model object fit using the \code{\link[DImodels:DI]{DI()}}
function from the \code{\link[DImodels:DImodels-package]{DImodels}}
package.}

\item{FG}{A character vector of same length as \code{prop} specifying the group
each variable belongs to.}

\item{npoints}{Number of points to be labelled in each plot, starting
with the most extreme (those points with the highest
absolute residuals or hat values).}

\item{cook_levels}{A numeric vector specifying levels of Cook's distance
at which to draw contours.}

\item{pie_radius}{A numeric value specifying the radius (in cm) for the
pie-glyphs.}

\item{pie_colours}{A character vector specifying the colours for the slices
within the pie-glyphs.}

\item{only_extremes}{A logical value indicating whether to show pie-glyphs
only for extreme observations (points with the highest
absolute residuals or hat values).}

\item{label_size}{A numeric value specifying the size of the labels
identifying extreme observations.}

\item{points_size}{A numeric value specifying the size of points (when
pie-glyphs not shown) shown in the plots.}

\item{plot}{A boolean variable indicating whether to create the plot or return
the prepared data instead. The default \code{TRUE} creates the plot while
\code{FALSE} would return the prepared data for plotting. Could be useful
if user wants to modify the data first and then create the plot.}

\item{nrow}{Number of rows in which to arrange the final plot.}

\item{ncol}{Number of columns in which to arrange the final plot.}
}
\value{
A ggmultiplot (ggplot if single plot is returned) class object or data-frame (if \code{plot = FALSE}).
}
\description{
This function returns regression diagnostics plots for a model with points
replaced by pie-glyphs making it easier to track various data points
in the plots. This could be useful in models with compositional predictors
to quickly identify any observations with unusual residuals, hat values, etc.
}
\examples{
library(DImodels)

## Load data
data(sim1)

## Fit model
mod1 <- lm(response ~ 0 + (p1 + p2 + p3 + p4)^2, data = sim1)

## Get diagnostics plot
## Recommend to store plot in a variable, to access individual plots later
diagnostics <- model_diagnostics(mod1, prop = c("p1", "p2", "p3", "p4"))
print(diagnostics)

## Access individual plots
print(diagnostics[[1]])
print(diagnostics[[4]])

## Change plot arrangement
\donttest{
model_diagnostics(mod1, prop = c("p1", "p2", "p3", "p4"),
                  which = c(1, 3), nrow = 2, ncol = 1)
}

## Show only extreme points as pie-glyphs to avoid overplotting
model_diagnostics(mod1, prop = c("p1", "p2", "p3", "p4"),
                  which = 2, npoints = 5, only_extremes = TRUE)

## If model is a DImodels object, the don't need to specify prop
DI_mod <- DI(y = "response", prop = c("p1", "p2", "p3", "p4"),
             DImodel = "FULL", data = sim1)
model_diagnostics(DI_mod, which = 1)

## Specify `plot = FALSE` to not create the plot but return the prepared data
head(model_diagnostics(DI_mod, which = 1, plot  = FALSE))
}
