% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/print_BKP.R, R/print_DKP.R
\name{print}
\alias{print}
\alias{print.BKP}
\alias{print.summary_BKP}
\alias{print.predict_BKP}
\alias{print.simulate_BKP}
\alias{print.DKP}
\alias{print.summary_DKP}
\alias{print.predict_DKP}
\alias{print.simulate_DKP}
\title{Print Methods for BKP and DKP Objects}
\usage{
\method{print}{BKP}(x, ...)

\method{print}{summary_BKP}(x, ...)

\method{print}{predict_BKP}(x, ...)

\method{print}{simulate_BKP}(x, ...)

\method{print}{DKP}(x, ...)

\method{print}{summary_DKP}(x, ...)

\method{print}{predict_DKP}(x, ...)

\method{print}{simulate_DKP}(x, ...)
}
\arguments{
\item{x}{An object of class \code{"BKP"} or \code{"DKP"}, or a derived object
such as \code{summary}, \code{predict}, or \code{simulate}.}

\item{...}{Additional arguments passed to the generic \code{print} method
(currently unused; included for S3 consistency).}
}
\value{
Invisibly returns the input object. Called for the side effect of
printing human-readable summaries to the console.
}
\description{
Provides formatted console output for fitted BKP/DKP model
objects, their summaries, predictions, and simulations. The following
specialized methods are supported:
\itemize{
\item \code{print.BKP}, \code{print.DKP} – display fitted model objects.
\item \code{print.summary_BKP}, \code{print.summary_DKP} – display
model summaries.
\item \code{print.predict_BKP}, \code{print.predict_DKP} – display
posterior predictive results.
\item \code{print.simulate_BKP}, \code{print.simulate_DKP} – display
posterior simulations.
}
}
\examples{
# ============================================================== #
# ========================= BKP Examples ======================= #
# ============================================================== #

#-------------------------- 1D Example ---------------------------
set.seed(123)

# Define true success probability function
true_pi_fun <- function(x) {
  (1 + exp(-x^2) * cos(10 * (1 - exp(-x)) / (1 + exp(-x)))) / 2
}

n <- 30
Xbounds <- matrix(c(-2,2), nrow=1)
X <- tgp::lhs(n = n, rect = Xbounds)
true_pi <- true_pi_fun(X)
m <- sample(100, n, replace = TRUE)
y <- rbinom(n, size = m, prob = true_pi)

# Fit BKP model
model1 <- fit_BKP(X, y, m, Xbounds=Xbounds)
print(model1)                    # fitted object
print(summary(model1))           # summary
print(predict(model1))           # predictions
print(simulate(model1, nsim=3))  # posterior simulations

#-------------------------- 2D Example ---------------------------
set.seed(123)

# Define 2D latent function and probability transformation
true_pi_fun <- function(X) {
  if(is.null(nrow(X))) X <- matrix(X, nrow=1)
  m <- 8.6928
  s <- 2.4269
  x1 <- 4*X[,1]- 2
  x2 <- 4*X[,2]- 2
  a <- 1 + (x1 + x2 + 1)^2 *
    (19- 14*x1 + 3*x1^2- 14*x2 + 6*x1*x2 + 3*x2^2)
  b <- 30 + (2*x1- 3*x2)^2 *
    (18- 32*x1 + 12*x1^2 + 48*x2- 36*x1*x2 + 27*x2^2)
  f <- log(a*b)
  f <- (f- m)/s
  return(pnorm(f))  # Transform to probability
}

n <- 100
Xbounds <- matrix(c(0, 0, 1, 1), nrow = 2)
X <- tgp::lhs(n = n, rect = Xbounds)
true_pi <- true_pi_fun(X)
m <- sample(100, n, replace = TRUE)
y <- rbinom(n, size = m, prob = true_pi)

# Fit BKP model
model2 <- fit_BKP(X, y, m, Xbounds=Xbounds)
print(model2)                    # fitted object
print(summary(model2))           # summary
print(predict(model2))           # predictions
print(simulate(model2, nsim=3))  # posterior simulations

# ============================================================== #
# ========================= DKP Examples ======================= #
# ============================================================== #

#-------------------------- 1D Example ---------------------------
set.seed(123)

# Define true class probability function (3-class)
true_pi_fun <- function(X) {
  p1 <- 1/(1+exp(-3*X))
  p2 <- (1 + exp(-X^2) * cos(10 * (1 - exp(-X)) / (1 + exp(-X)))) / 2
  return(matrix(c(p1/2, p2/2, 1 - (p1+p2)/2), nrow = length(p1)))
}

n <- 30
Xbounds <- matrix(c(-2, 2), nrow = 1)
X <- tgp::lhs(n = n, rect = Xbounds)
true_pi <- true_pi_fun(X)
m <- sample(150, n, replace = TRUE)

# Generate multinomial responses
Y <- t(sapply(1:n, function(i) rmultinom(1, size = m[i], prob = true_pi[i, ])))

# Fit DKP model
model1 <- fit_DKP(X, Y, Xbounds = Xbounds)
print(model1)                    # fitted object
print(summary(model1))           # summary
print(predict(model1))           # predictions
print(simulate(model1, nsim=3))  # posterior simulations


#-------------------------- 2D Example ---------------------------
set.seed(123)

# Define latent function and transform to 3-class probabilities
true_pi_fun <- function(X) {
  if (is.null(nrow(X))) X <- matrix(X, nrow = 1)
  m <- 8.6928; s <- 2.4269
  x1 <- 4 * X[,1] - 2
  x2 <- 4 * X[,2] - 2
  a <- 1 + (x1 + x2 + 1)^2 *
    (19 - 14*x1 + 3*x1^2 - 14*x2 + 6*x1*x2 + 3*x2^2)
  b <- 30 + (2*x1 - 3*x2)^2 *
    (18 - 32*x1 + 12*x1^2 + 48*x2 - 36*x1*x2 + 27*x2^2)
  f <- (log(a*b)- m)/s
  p1 <- pnorm(f) # Transform to probability
  p2 <- sin(pi * X[,1]) * sin(pi * X[,2])
  return(matrix(c(p1/2, p2/2, 1 - (p1+p2)/2), nrow = length(p1)))
}

n <- 100
Xbounds <- matrix(c(0, 0, 1, 1), nrow = 2)
X <- tgp::lhs(n = n, rect = Xbounds)
true_pi <- true_pi_fun(X)
m <- sample(150, n, replace = TRUE)

# Generate multinomial responses
Y <- t(sapply(1:n, function(i) rmultinom(1, size = m[i], prob = true_pi[i, ])))

# Fit DKP model
model2 <- fit_DKP(X, Y, Xbounds = Xbounds)
print(model2)                    # fitted object
print(summary(model2))           # summary
print(predict(model2))           # predictions
print(simulate(model2, nsim=3))  # posterior simulations

}
\references{
Zhao J, Qing K, Xu J (2025). \emph{BKP: An R Package for Beta
Kernel Process Modeling}.  arXiv.
https://doi.org/10.48550/arXiv.2508.10447.
}
\seealso{
\code{\link{fit_BKP}}, \code{\link{fit_DKP}} for model fitting;
\code{\link{summary.BKP}}, \code{\link{summary.DKP}} for model summaries;
\code{\link{predict.BKP}}, \code{\link{predict.DKP}} for posterior
prediction; \code{\link{simulate.BKP}}, \code{\link{simulate.DKP}} for
posterior simulations.
}
\keyword{BKP}
\keyword{DKP}
